package ru.yandex.autotests.direct.api.bidmodifiers.toggle;

import com.yandex.direct.api.v5.bidmodifiers.BidModifierToggleTypeEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import org.apache.commons.lang3.StringUtils;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersFeatures;
import ru.yandex.autotests.direct.api.bidmodifiers.BidModifiersLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.BidModifierToggleMap;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.JavaOrPerlToggleExpectedResult;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.ToggleExpectedResult;
import ru.yandex.autotests.directapi.model.api5.bidmodifiers.ToggleRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlApi5Error;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 10.08.15
 */
@Aqua.Test
@Features(BidModifiersFeatures.TOGGLE)
@Stories(ApiStories.UNITS)
@Tag(TagDictionary.TRUNK)
@Description("Списание баллов при отключении наборов корректировок ставок")
public class ToggleBidModifiersUnitsTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());
    private static final String client = BidModifiersLogins.CLIENT_UNITS_TOGGLE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final int TOGGLE_COST = 15;
    private static final int ITEM_COST_ON_TOGGLE = 0;
    private static final int ITEM_COST_ON_TOGGLE_FAULT = 0;
    private static final int GENERAL_FAULT_COST = 50;

    private static final Long INVALID_ID = 123456L;
    private static final Long INVALID_ID_2 = 123457L;

    private Long groupID;
    private static Long campaignID;

    @BeforeClass
    public static void prepareCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
    }

    @Before
    public void prepareGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
    }

    @Test
    @Description("Отключение одиночного набора корректировок")
    public void toggleSingleBidModifier() {
        api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupID);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.bidModifiersSteps()
                .toggleBidModifiersGroup(groupID, BidModifierToggleTypeEnum.DEMOGRAPHICS_ADJUSTMENT, YesNoEnum.NO);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость отключения одного набора корректировок", unitsBefore - unitsAfter,
                equalTo(TOGGLE_COST + ITEM_COST_ON_TOGGLE));
    }

    @Test
    @Description("Отключение нескольких наборов корректировок")
    public void toggleSeveralBidModifiersTest() {
        Integer conditionId = api.userSteps.retargetingSteps().addRandomRetargetingCondition(client);
        api.userSteps.bidModifiersSteps().addBidModifierRetargetingToGroup(conditionId.longValue(), groupID);
        api.userSteps.bidModifiersSteps().addBidModifierDemographicsToCampaign(campaignID.intValue());

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.bidModifiersSteps().bidModifiersToggle(
                new BidModifierToggleMap()
                        .withCampaignId(campaignID)
                        .withEnabled(YesNoEnum.NO)
                        .withType(BidModifierToggleTypeEnum.DEMOGRAPHICS_ADJUSTMENT),
                new BidModifierToggleMap()
                        .withAdGroupId(groupID)
                        .withEnabled(YesNoEnum.NO)
                        .withType(BidModifierToggleTypeEnum.RETARGETING_ADJUSTMENT));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость отключения двух наборов корректировок", unitsBefore - unitsAfter,
                equalTo(TOGGLE_COST + ITEM_COST_ON_TOGGLE * 2));
    }

    @Test
    @Description("Отключение одного невалидного набора корректировок")
    public void toggleInvalidBidModifierTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.bidModifiersSteps().shouldGetToggleResultOn(
                null,
                new ToggleRequestMap().withBidModifierToggleItems(new BidModifierToggleMap()
                        .withEnabled(YesNoEnum.NO)
                        .withType(BidModifierToggleTypeEnum.DEMOGRAPHICS_ADJUSTMENT)
                        .withAdGroupId(INVALID_ID)),
                JavaOrPerlToggleExpectedResult.java(
                        ToggleExpectedResult.errors(new Notification(8800, Api5ErrorDetailsJava.GROUP_NOT_FOUND,
                                INVALID_ID)))
                        .perl(
                                ToggleExpectedResult.errors(new Notification(8800, Api5ErrorDetails.GROUP_NOT_FOUND))));

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при отключении набора корректировок", unitsBefore - unitsAfter,
                equalTo(TOGGLE_COST + ITEM_COST_ON_TOGGLE_FAULT));
    }

    @Test
    @Description("Отключение двух невалидных наборов корректировок")
    public void setFewInvalidBidModifiersTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.bidModifiersSteps().shouldGetToggleResultOn(
                null,
                new ToggleRequestMap().withBidModifierToggleItems(
                        new BidModifierToggleMap()
                                .withAdGroupId(INVALID_ID)
                                .withEnabled(YesNoEnum.NO)
                                .withType(BidModifierToggleTypeEnum.DEMOGRAPHICS_ADJUSTMENT),
                        new BidModifierToggleMap()
                                .withCampaignId(INVALID_ID_2)
                                .withEnabled(YesNoEnum.NO)
                                .withType(BidModifierToggleTypeEnum.RETARGETING_ADJUSTMENT)
                ),
                JavaOrPerlToggleExpectedResult.java(
                        ToggleExpectedResult.errors(new Notification(8800, Api5ErrorDetailsJava.GROUP_NOT_FOUND,
                                INVALID_ID)))
                        .perl(
                                ToggleExpectedResult.errors(new Notification(8800, Api5ErrorDetails.GROUP_NOT_FOUND))),
                JavaOrPerlToggleExpectedResult.java(
                        ToggleExpectedResult.errors(new Notification(8800, Api5ErrorDetailsJava.CAMPAIGN_NOT_FOUND,
                                INVALID_ID_2)))
                        .perl(
                                ToggleExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND))
                        ));

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при отключении двух наборов корректировок", unitsBefore - unitsAfter,
                equalTo(TOGGLE_COST + ITEM_COST_ON_TOGGLE_FAULT * 2));
    }

    @Test
    @Description("Отключение одного корректного и одного невалидного набора корректировок")
    public void setOneValidAndOneInvalidBidModifiersTest() {
        api.userSteps.bidModifiersSteps().addBidModifierDemographicsToGroup(groupID);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.bidModifiersSteps().shouldGetToggleResultOn(
                null,
                new ToggleRequestMap().withBidModifierToggleItems(
                        new BidModifierToggleMap()
                                .withAdGroupId(groupID)
                                .withEnabled(YesNoEnum.NO)
                                .withType(BidModifierToggleTypeEnum.DEMOGRAPHICS_ADJUSTMENT)
                        ,
                        new BidModifierToggleMap()
                                .withCampaignId(INVALID_ID_2)
                                .withEnabled(YesNoEnum.NO)
                                .withType(BidModifierToggleTypeEnum.RETARGETING_ADJUSTMENT)
                ),
                JavaOrPerlToggleExpectedResult.both(ToggleExpectedResult.successGroup(groupID)),
                JavaOrPerlToggleExpectedResult.java(
                        ToggleExpectedResult.errors(new Notification(8800, Api5ErrorDetailsJava.CAMPAIGN_NOT_FOUND,
                                INVALID_ID_2)))
                        .perl(
                                ToggleExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND))
                        ));

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при отключении нескольких наборов корректировок",
                unitsBefore - unitsAfter, equalTo(TOGGLE_COST + ITEM_COST_ON_TOGGLE + ITEM_COST_ON_TOGGLE_FAULT));
    }

    @Test
    @Description("Невалидный запрос на отключение набора корректировок")
    public void generalErrorOnSetBidModifierTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.TOGGLE,
                new ToggleRequestMap(),
                JavaOrPerlApi5Error.java(
                        new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                                StringUtils.capitalize(ToggleRequestMap.BID_MODIFIER_TOGGLE_ITEMS)))
                        .perl(
                                new Api5Error(8000,
                                        Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                                        "params",
                                        StringUtils.capitalize(ToggleRequestMap.BID_MODIFIER_TOGGLE_ITEMS))));

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость общей ошибки при отключении набора корректировок", unitsBefore - unitsAfter,
                equalTo(GENERAL_FAULT_COST));
    }

    @Test
    @Description("Недостаточное колличество баллов при отключении набора корректировок")
    public void notEnoughUnitsOnSetBidModifierTest() {
        api.userSteps.bidModifiersSteps().addBidModifierMobileToGroup(groupID);

        log.info("Обнулим колличество доступных пользователю баллов");
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(client, unitsBefore);
        unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assumeThat("доступные баллы пользователя", unitsBefore, equalTo(0));

        api.userSteps.bidModifiersSteps().shouldGetBidModifierErrorOn(
                Action.TOGGLE,
                new ToggleRequestMap().withBidModifierToggleItems(
                        new BidModifierToggleMap()
                                .withAdGroupId(groupID)
                                .withEnabled(YesNoEnum.NO)
                                .withType(BidModifierToggleTypeEnum.DEMOGRAPHICS_ADJUSTMENT)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("доступные баллы пользователя после ошибки", unitsAfter, equalTo(0));
    }
}
