package ru.yandex.autotests.direct.api.bids.get;

import java.util.Arrays;
import java.util.Collection;
import java.util.function.Function;

import javax.xml.bind.JAXBElement;

import com.yandex.direct.api.v5.bids.BidFieldEnum;
import com.yandex.direct.api.v5.bids.BidGetItem;
import com.yandex.direct.api.v5.bids.GetResponse;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bids.BidsFeatures;
import ru.yandex.autotests.direct.api.bids.BidsLogins;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.bids.BidsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bids.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.TestCaseId;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by gerdler on 03.04.17.
 * https://st.yandex-team.ru/DIRECT-64666
 */
@Aqua.Test
@Features(BidsFeatures.GET)
@Issue("https://st.yandex-team.ru/DIRECT-64386")
@Description("Проверка получения nil для nillable-полей автотаргетинга")
@RunWith(Parameterized.class)
public class BidsGetAutotargetingNillableFieldsTest {
    private static final String CLIENT = BidsLogins.CLIENT_FOR_RUB;
    private static long campaignId;
    private static long adGroupId;
    private static Long autoTargetingKeywordId;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(0)
    public BidFieldEnum bidField;

    @Parameterized.Parameter(1)
    public Function<BidGetItem, JAXBElement> fieldGetter;

    @Parameterized.Parameter(2)
    public boolean expectedResult;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> testData() {
        return Arrays.asList(new Object[][]{
                {BidFieldEnum.STRATEGY_PRIORITY, (Function<BidGetItem, JAXBElement>) BidGetItem::getStrategyPriority, true},
                {BidFieldEnum.CONTEXT_COVERAGE, (Function<BidGetItem, JAXBElement>) BidGetItem::getContextCoverage, true},
                {BidFieldEnum.CURRENT_SEARCH_PRICE, (Function<BidGetItem, JAXBElement>) BidGetItem::getCurrentSearchPrice, true},
        });
    }

    @BeforeClass
    public static void prepare() {
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        autoTargetingKeywordId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);
        DirectJooqDbSteps dbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT);
        dbSteps.bidsBaseSteps().updateBidsBase(
                dbSteps.bidsBaseSteps().getBidsBaseByBidId(autoTargetingKeywordId).setAutobudgetpriority(null)
        );
    }

    @Test
    @TestCaseId("2706")
    public void hiddenFieldShouldNotBeReturned() {
        GetResponse response = api.userSteps.bidsSteps().bidsGet(
                new GetRequestMap().withSelectionCriteria(
                        new BidsSelectionCriteriaMap().withKeywordIds(autoTargetingKeywordId)
                ).withFieldNames(bidField)
        );

        assumeThat("вернулся автотаргетинг", response.getBids(), not(empty()));
        BidGetItem bidGetItem = response.getBids().get(0);
        assertThat("получили значение nil", fieldGetter.apply(bidGetItem).isNil(), equalTo(expectedResult));
    }
}
