package ru.yandex.autotests.direct.api.bids.get;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.bids.BidFieldEnum;
import com.yandex.direct.api.v5.bids.BidGetItem;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bids.BidsFeatures;
import ru.yandex.autotests.direct.api.bids.BidsKeywords;
import ru.yandex.autotests.direct.api.bids.BidsLogins;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.bids.BidsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bids.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignNetworkStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignSearchStrategyAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by hmepas on 21.12.16
 */
@Aqua.Test
@Features(BidsFeatures.GET)
@Description("Проверяем правильную работу получения ContextCoverage для кампании с показами на сети и без в одном запросе")
@Issue("https://st.yandex-team.ru/DIRECT-61741")
public class BidsGetContextCoverageCheckWithThreeCampaignsInOneRequest {

    private static final String LOGIN = BidsLogins.BIDS_CLIENT;
    private static final String KEYWORD = BidsKeywords.WITH_POKAZOMETER_DATA;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    private static Long adGroupIdWithContext;
    private static Long adGroupIdWithContextNoAds;
    private static Long adGroupIdWithoutContext;
    private static Long adGroupIdWithContext2;
    private static Map<Long, BidGetItem> bidsGetItemByAdGroupId;

    @BeforeClass
    @Step("Подготовка данных для теста")
    public static void createKeywords() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        // порядок важен, нужно кампания с показами на сети, как первая в списке так и не первая, поэтому
        // создаем две, одну в конце
        // https://st.yandex-team.ru/DIRECT-62086
        Long campaignIdWithContext = api.as(LOGIN).userSteps.campaignSteps().addDefaultTextCampaignWithIndependentStrategies();

        adGroupIdWithContext = api.userSteps.adGroupsSteps().addDefaultGroup(campaignIdWithContext);
        api.userSteps.adsSteps().addDefaultTextAd(adGroupIdWithContext);
        api.userSteps.keywordsSteps().addKeyword(adGroupIdWithContext, KEYWORD);

        adGroupIdWithContextNoAds = api.userSteps.adGroupsSteps().addDefaultGroup(campaignIdWithContext);
        api.userSteps.keywordsSteps().addKeyword(adGroupIdWithContextNoAds, KEYWORD);

        Long campaignIdWithoutContext = api.as(LOGIN).userSteps.campaignSteps().addDefaultTextCampaignWithStrategies(
                new TextCampaignSearchStrategyAddMap().defaultWbMaximumClicks(Currency.RUB),
                new TextCampaignNetworkStrategyAddMap().defaultNetworkDefault()
        );
        adGroupIdWithoutContext = api.userSteps.adGroupsSteps().addDefaultGroup(campaignIdWithoutContext);
        api.userSteps.adsSteps().addDefaultTextAd(adGroupIdWithoutContext);
        api.userSteps.keywordsSteps().addKeyword(adGroupIdWithoutContext, KEYWORD);

        Long campaignIdWithContext2 = api.as(LOGIN).userSteps.campaignSteps().addDefaultTextCampaignWithIndependentStrategies();

        adGroupIdWithContext2 = api.userSteps.adGroupsSteps().addDefaultGroup(campaignIdWithContext2);
        api.userSteps.adsSteps().addDefaultTextAd(adGroupIdWithContext2);
        api.userSteps.keywordsSteps().addKeyword(adGroupIdWithContext2, KEYWORD);


        List<BidGetItem> bidsGetItems =
                api.userSteps.bidsSteps().bidsGet(new GetRequestMap().withSelectionCriteria(
                        new BidsSelectionCriteriaMap().withAdGroupIds(adGroupIdWithContext, adGroupIdWithContextNoAds, adGroupIdWithoutContext, adGroupIdWithContext2))
                                .withFieldNames(BidFieldEnum.AD_GROUP_ID, BidFieldEnum.CONTEXT_COVERAGE),
                        LOGIN).getBids();

        assumeThat("Вернулись данные по четырем фразам", bidsGetItems, hasSize(4));
        bidsGetItemByAdGroupId = bidsGetItems.stream().collect(Collectors.toMap(k -> k.getAdGroupId(), v -> v));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("2648")
    public void hasContextCoverageForFirstCampaignId() {
        assertThat("в ответе присутствует ContextCoverage для первой в списке кампании с раздельным управлением ставками на сети",
                !bidsGetItemByAdGroupId.get(adGroupIdWithContext).getContextCoverage().isNil());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("2648")
    public void hasContextCoverageForNonFirstCampaignId() {
        // https://st.yandex-team.ru/DIRECT-62086
        assertThat("в ответе присутствует ContextCoverage для не первой в списке кампании с раздельным управлением ставками на сети",
                !bidsGetItemByAdGroupId.get(adGroupIdWithContext2).getContextCoverage().isNil());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("2647")
    public void contextCoverageIsNull() {
        assertThat("ContextCoverage null для кампании без независимого управления ставками на сети",
                bidsGetItemByAdGroupId.get(adGroupIdWithoutContext).getContextCoverage().isNil());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("2649")
    public void contextCoverageIsNullForAdGroupWithNoAds() {
        assertThat("ContextCoverage null для кампании без независимого управления ставками на сети",
                bidsGetItemByAdGroupId.get(adGroupIdWithContextNoAds).getContextCoverage().isNil());
    }
}
