package ru.yandex.autotests.direct.api.bids.get;

import java.util.List;

import com.yandex.direct.api.v5.bids.BidFieldEnum;
import com.yandex.direct.api.v5.bids.ContextCoverageItem;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bids.BidsFeatures;
import ru.yandex.autotests.direct.api.bids.BidsKeywords;
import ru.yandex.autotests.direct.api.bids.BidsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.bids.BidsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bids.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.greaterThanOrEqualTo;
import static org.hamcrest.Matchers.lessThanOrEqualTo;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by semkagtn on 01.09.15.
 * https://st.yandex-team.ru/TESTIRT-6933
 */
@Aqua.Test
@Features(BidsFeatures.GET)
@Description("Проверка минимального и максимального значения Probability в массиве ContextCoverage")
@Issue("https://st.yandex-team.ru/DIRECT-45134")
public class BidsGetContextCoverageProbabilityMinAndMaxValueTest {

    private static final String LOGIN = BidsLogins.BIDS_CLIENT;

    private static final int MIN_PROBABILITY_LOWER = 0;
    private static final int MIN_PROBABILITY_UPPER = 20;

    private static final int MAX_PROBABILITY_LOWER = 100;
    private static final int MAX_PROBABILITY_UPPER = 100;

    private static final String KEYWORD = BidsKeywords.WITH_POKAZOMETER_DATA;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    private static long keywordId;

    @BeforeClass
    public static void createKeyword() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaignWithIndependentStrategies();

        long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        keywordId = api.userSteps.keywordsSteps().addKeyword(adGroupId, KEYWORD);
    }

    @Test
    @Description("Проверка минимального значения Probability в массиве ContextCoverage")
    @ru.yandex.qatools.allure.annotations.TestCaseId("1218")
    public void checkMinimumProbabilityValue() {
        List<ContextCoverageItem> contextCoverageItems = api.userSteps.bidsSteps().bidsGet(new GetRequestMap().withSelectionCriteria(
                        new BidsSelectionCriteriaMap().withKeywordIds(keywordId))
                        .withFieldNames(BidFieldEnum.CONTEXT_COVERAGE),
                LOGIN).getBids().get(0).getContextCoverage().getValue().getItems();
        assumeThat("вернулся не пустой массив ContextCoverage", contextCoverageItems, not(empty()));

        int minProbability = contextCoverageItems.stream()
                .map(x -> x.getProbability().intValue())
                .min(Integer::compareTo)
                .orElse(-1);

        assertThat("минимальное значение Probability находится в допустимых пределах", minProbability,
                allOf(greaterThanOrEqualTo(MIN_PROBABILITY_LOWER), lessThanOrEqualTo(MIN_PROBABILITY_UPPER)));
    }

    @Test
    @Description("Проверка максимального значения Probability в массиве ContextCoverage")
    @ru.yandex.qatools.allure.annotations.TestCaseId("1217")
    public void checkMaximumProbabilityValue() {
        List<ContextCoverageItem> contextCoverageItems = api.userSteps.bidsSteps().bidsGet(new GetRequestMap().withSelectionCriteria(
                        new BidsSelectionCriteriaMap().withKeywordIds(keywordId))
                        .withFieldNames(BidFieldEnum.CONTEXT_COVERAGE),
                LOGIN).getBids().get(0).getContextCoverage().getValue().getItems();
        assumeThat("вернулся не пустой массив ContextCoverage", contextCoverageItems, not(empty()));

        int maxProbability = contextCoverageItems.stream()
                .map(x -> x.getProbability().intValue())
                .max(Integer::compareTo)
                .orElse(-1);

        assertThat("максимальное значение Probability находится в допустимых пределах", maxProbability,
                allOf(greaterThanOrEqualTo(MAX_PROBABILITY_LOWER), lessThanOrEqualTo(MAX_PROBABILITY_UPPER)));
    }
}
