package ru.yandex.autotests.direct.api.bids.get;

import java.math.BigDecimal;
import java.util.List;
import java.util.stream.Collectors;

import com.google.common.collect.Ordering;
import com.yandex.direct.api.v5.bids.BidFieldEnum;
import com.yandex.direct.api.v5.bids.ContextCoverageItem;
import com.yandex.direct.api.v5.bids.GetResponse;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bids.BidsFeatures;
import ru.yandex.autotests.direct.api.bids.BidsKeywords;
import ru.yandex.autotests.direct.api.bids.BidsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.bids.BidsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bids.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by ginger on 20.08.15.
 *https://st.yandex-team.ru/TESTIRT-6760
 */
@Aqua.Test
@Features(BidsFeatures.GET)
@Description("Проверка сортировки Probability, Price из ContextCoverage.")
@Issue("https://st.yandex-team.ru/DIRECT-45134")
public class BidsGetContextCoverageSortTest {

    private static final String KEYWORD = BidsKeywords.WITH_POKAZOMETER_DATA;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(BidsLogins.BIDS_CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();
    private static Long keywordID;
    private static GetResponse getResponse;
    private static List<ContextCoverageItem> gotContextCoverage;

    @Rule
    public Trashman trashman = new Trashman(api);

    @BeforeClass
    public static void createKeyword() {
        Long campaignID = api.as(BidsLogins.BIDS_CLIENT).userSteps.campaignSteps().addDefaultTextCampaignWithIndependentStrategies();

        Long adGroupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        api.userSteps.adsSteps().addDefaultTextAd(adGroupID);
        keywordID = api.userSteps.keywordsSteps().addKeyword(adGroupID, KEYWORD);
        getResponse  = api.userSteps.bidsSteps().bidsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new BidsSelectionCriteriaMap().withKeywordIds(keywordID)
                        )
                        .withFieldNames(BidFieldEnum.CONTEXT_COVERAGE),
                BidsLogins.BIDS_CLIENT
        );
        assumeThat("вернулась 1 ставка", getResponse.getBids(), hasSize(1));
        assumeThat("в ответе присутствует ContextCoverage",
                getResponse.getBids().get(0).getContextCoverage().getValue().getItems(),
                hasSize(equalTo(3)));
        gotContextCoverage = getResponse.getBids().get(0).getContextCoverage().getValue().getItems();
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("1220")
    public void checkSortingOfFieldProbability() {
        List<BigDecimal> probability = gotContextCoverage.stream()
                .map(ContextCoverageItem::getProbability)
                .collect(Collectors.toList());
        assertThat("значения Probability убывают", Ordering.natural().isOrdered(probability), equalTo(false));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("1221")
    public void checkSortingOfFieldPrice() {
        List<Long> prices = gotContextCoverage.stream()
                .map(ContextCoverageItem::getPrice)
                .collect(Collectors.toList());
        assertThat("значения Price убывают", Ordering.natural().isStrictlyOrdered(prices), equalTo(false));
    }
}
