package ru.yandex.autotests.direct.api.bids.get;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.function.Function;

import com.yandex.direct.api.v5.bids.BidFieldEnum;
import com.yandex.direct.api.v5.bids.BidGetItem;
import com.yandex.direct.api.v5.bids.GetResponse;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bids.BidsFeatures;
import ru.yandex.autotests.direct.api.bids.BidsLogins;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.bids.BidsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bids.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by onotole on 11.01.17.
 * https://st.yandex-team.ru/TESTIRT-10840
 */
@Aqua.Test
@Features(BidsFeatures.GET)
@Stories(ApiStories.STATUS)
@Issue("https://st.yandex-team.ru/DIRECT-61797")
@Description("Проверка возвращаемых значений для группы, содержащей только графические объявления")
@RunWith(Parameterized.class)
public class BidsGetGraphicsOnlyTest {
    private static final String CLIENT = BidsLogins.CLIENT_FOR_RUB;
    private static long campaignId;
    private static long adGroupId;
    private static String adImageHash;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(0)
    public BidFieldEnum bidField;

    @Parameterized.Parameter(1)
    public Function<BidGetItem, List> getField;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> testData() {
        return Arrays.asList(new Object[][] {
                {BidFieldEnum.COMPETITORS_BIDS, (Function<BidGetItem, List>) BidGetItem::getCompetitorsBids},
                {BidFieldEnum.SEARCH_PRICES, (Function<BidGetItem, List>) BidGetItem::getSearchPrices},
                {BidFieldEnum.AUCTION_BIDS, (Function<BidGetItem, List>) BidGetItem::getAuctionBids}
        });
    }

    @BeforeClass
    public static void prepare() {
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        adImageHash = api.userSteps.imagesStepsV5().addImageAdAdImage(1);
        api.userSteps.adsSteps().addDefaultTextImageAd(adGroupId, adImageHash);
        api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("2651")
    public void hiddenFieldShouldNotBeReturned() {
        GetResponse response = api.userSteps.bidsSteps().bidsGet(
                new GetRequestMap().withSelectionCriteria(
                        new BidsSelectionCriteriaMap()
                                .withAdGroupIds(adGroupId))
                        .withFieldNames(bidField));

        assumeThat("фраза вернулась", response.getBids(), not(empty()));
        BidGetItem bidGetItem = response.getBids().get(0);
        assertThat("не отображаемое поле не вернулось", getField.apply(bidGetItem).size(), equalTo(0));
    }
}
