package ru.yandex.autotests.direct.api.bids.get;

import java.math.BigDecimal;
import java.util.Collections;
import ru.yandex.qatools.allure.annotations.TestCaseId;

import com.yandex.direct.api.v5.bids.BidFieldEnum;
import com.yandex.direct.api.v5.bids.GetResponse;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bids.BidsFeatures;
import ru.yandex.autotests.direct.api.bids.BidsLogins;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.api5.bids.BidsGetItemMap;
import ru.yandex.autotests.directapi.model.api5.bids.BidsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bids.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsNull.notNullValue;
import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

@Aqua.Test
@Features(BidsFeatures.GET)
@Description("Проверка чтения нулевых ставок")
@Issue("https://st.yandex-team.ru/DIRECT-76721")
public class BidsGetZeroPricesTest {
    private static final String CLIENT = BidsLogins.BIDS_CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(Logins.SUPER_LOGIN);
    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    private Long keywordId;

    @Before
    public void prepare() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        api.userSteps.clientFakeSteps().enableAPI(CLIENT);
        int shard = api.userSteps.clientFakeSteps().getUserShard(CLIENT);

        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(CLIENT);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID, CLIENT);
        api.userSteps.adsSteps().addDefaultTextAd(adGroupId, CLIENT);
        keywordId = api.userSteps.keywordsSteps().addDefaultKeyword(CLIENT, adGroupId);
        api.userSteps.getDirectJooqDbSteps().useShard(shard).bidsSteps().setBidsPrice(keywordId, BigDecimal.ZERO);
        api.userSteps.getDirectJooqDbSteps().useShard(shard).bidsSteps()
                .setBidsPriceContext(keywordId, BigDecimal.ZERO);
    }

    @Test
    @TestCaseId("2817")
    public void getBidsWithZeroPrices() {
        Long minBid = MoneyCurrency.get(Currency.RUB).getMinPrice().bidLong().longValue();
        GetResponse getResponse = api.userSteps.bidsSteps().bidsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new BidsSelectionCriteriaMap().withKeywordIds(keywordId)
                        )
                        .withFieldNames(
                                BidFieldEnum.BID,
                                BidFieldEnum.CONTEXT_BID
                        ),
                CLIENT
        );
        assertThat("вернулись корректные значения Bid",
                getResponse.getBids(),
                beanDifferV5(
                        Collections
                                .singletonList(new BidsGetItemMap().withBid(minBid).withContextBid(minBid).getBean()))
        );
    }

    @Test
    @TestCaseId("2816")
    public void getContextCoverage() {
        GetResponse getResponse = api.userSteps.bidsSteps().bidsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new BidsSelectionCriteriaMap().withKeywordIds(keywordId)
                        )
                        .withFieldNames(
                                BidFieldEnum.CONTEXT_COVERAGE
                        ),
                CLIENT
        );
        assertThat("вернулось Значение ContextCoverage",
                getResponse.getBids().get(0).getContextCoverage(),
                is(notNullValue())
        );
    }

}
