package ru.yandex.autotests.direct.api.bids.get;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import com.yandex.direct.api.v5.bids.BidFieldEnum;
import com.yandex.direct.api.v5.bids.BidGetItem;
import com.yandex.direct.api.v5.bids.GetResponse;
import com.yandex.direct.api.v5.general.ServingStatusEnum;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bids.BidsFeatures;
import ru.yandex.autotests.direct.api.bids.BidsLogins;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.bids.BidsGetItemMap;
import ru.yandex.autotests.directapi.model.api5.bids.BidsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bids.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by onotole on 26.12.16.
 * https://st.yandex-team.ru/TESTIRT-10840
 */
@Aqua.Test
@Features(BidsFeatures.GET)
@Stories(ApiStories.STATUS)
@Issue("https://st.yandex-team.ru/DIRECT-61797")
@Description("Проверка фильтрации по полю ServingStatus")
@RunWith(Parameterized.class)
public class GetBsRarelyLoadedFilteredTest {
    private static final String CLIENT = BidsLogins.CLIENT_FOR_RUB;
    private static Long campaignId;
    private static Long adGroupRarelyServed;
    private static Long adGroupEligible;
    private static Long keywordIdEligible;
    private static Long relevanceMatchKeywordIdEligible;
    private static Long keywordIdRarelyServed;
    private static Long relevanceMatchKeywordIdRarelyServed;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    public static void setUp() {
        DirectJooqDbSteps jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();

        adGroupEligible = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        keywordIdEligible = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupEligible);
        relevanceMatchKeywordIdEligible = api.userSteps.keywordsSteps().addAutotargeting(adGroupEligible);
        jooqDbSteps.adGroupsSteps().setBsRarelyLoaded(adGroupEligible, false);

        adGroupRarelyServed = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        keywordIdRarelyServed = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupRarelyServed);
        relevanceMatchKeywordIdRarelyServed = api.userSteps.keywordsSteps().addAutotargeting(adGroupRarelyServed);
        jooqDbSteps.adGroupsSteps().setBsRarelyLoaded(adGroupRarelyServed, true);
    }

    @Parameterized.Parameter(0)
    public ServingStatusEnum servingStatus;

    @Parameterized.Parameter(1)
    public List<BidGetItem> expectedResult;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> testDate() {
        setUp();
        return Arrays.asList(new Object[][]{
                {
                        ServingStatusEnum.RARELY_SERVED,
                        Arrays.asList(
                                new BidsGetItemMap().withKeywordId(keywordIdRarelyServed).getBean(),
                                new BidsGetItemMap().withKeywordId(relevanceMatchKeywordIdRarelyServed).getBean()
                        )
                },
                {
                        ServingStatusEnum.ELIGIBLE,
                        Arrays.asList(
                                new BidsGetItemMap().withKeywordId(keywordIdEligible).getBean(),
                                new BidsGetItemMap().withKeywordId(relevanceMatchKeywordIdEligible).getBean()
                        )
                }
        });
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("2652")
    public void checkFilterCampaignIds() {
        GetResponse response = api.userSteps.bidsSteps().bidsGet(
                new GetRequestMap().withSelectionCriteria(
                        new BidsSelectionCriteriaMap()
                                .withCampaignIds(campaignId)
                                .withServingStatuses(servingStatus))
                        .withFieldNames(BidFieldEnum.KEYWORD_ID));
        assertThat("количество элементов и значения полей в ответе соответствуют ожидаемым",
                response.getBids(), beanDifferV5(expectedResult));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("2653")
    public void checkFilterAdGroupIds() {
        GetResponse response = api.userSteps.bidsSteps().bidsGet(
                new GetRequestMap().withSelectionCriteria(
                        new BidsSelectionCriteriaMap()
                                .withAdGroupIds(adGroupEligible, adGroupRarelyServed)
                                .withServingStatuses(servingStatus))
                        .withFieldNames(BidFieldEnum.KEYWORD_ID));
        assertThat("количество элементов и значения полей в ответе соответствуют ожидаемым",
                response.getBids(), beanDifferV5(expectedResult));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("2654")
    public void checkFilterKeywordIds() {
        GetResponse response = api.userSteps.bidsSteps().bidsGet(
                new GetRequestMap().withSelectionCriteria(
                        new BidsSelectionCriteriaMap()
                                .withKeywordIds(keywordIdEligible, keywordIdRarelyServed, relevanceMatchKeywordIdEligible, relevanceMatchKeywordIdRarelyServed)
                                .withServingStatuses(servingStatus))
                        .withFieldNames(BidFieldEnum.KEYWORD_ID));
        assertThat("количество элементов и значения полей в ответе соответствуют ожидаемым",
                response.getBids(), beanDifferV5(expectedResult));
    }
}
