package ru.yandex.autotests.direct.api.bids.set;

import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.bids.BidFieldEnum;
import com.yandex.direct.api.v5.bids.GetResponse;
import com.yandex.direct.api.v5.bids.ObjectFactory;
import com.yandex.direct.api.v5.general.PriorityEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bids.BidsFeatures;
import ru.yandex.autotests.direct.api.bids.BidsLogins;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.bids.BidExpectedResult;
import ru.yandex.autotests.directapi.model.api5.bids.BidSetItemMap;
import ru.yandex.autotests.directapi.model.api5.bids.BidsGetItemMap;
import ru.yandex.autotests.directapi.model.api5.bids.BidsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bids.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.bids.SetRequestMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignNetworkStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignSearchStrategyAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.TestCaseId;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by gerdler on 11.04.17.
 */
@Aqua.Test
@Features(BidsFeatures.SET)
@RunWith(Parameterized.class)
public class SetStrategyPriorityPositiveTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(BidsLogins.BIDS_CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    @Parameterized.Parameter(0)
    public PriorityEnum priority;

    @Parameterized.Parameters(name = "StrategyPriority = {0}")
    public static Collection testData() {
        Object[][] data = new Object[][]{
                {PriorityEnum.HIGH},
                {PriorityEnum.NORMAL},
                {PriorityEnum.LOW},
        };
        return Arrays.asList(data);
    }

    private static Long campaignId;
    private Long adGroupId;

    @BeforeClass
    @Step("Подготовка данных для теста")
    public static void createCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(BidsLogins.BIDS_CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaignWithStrategies(
                new TextCampaignSearchStrategyAddMap().defaultWbMaximumClicks(Currency.RUB),
                new TextCampaignNetworkStrategyAddMap().defaultNetworkDefault()
        );
    }

    @Before
    public void createAdGroup() {
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
    }

    @Test
    @TestCaseId("2731")
    public void setStrategyPriorityForKeyword() {
        Long keywordId = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
        api.userSteps.bidsSteps().shouldGetBidResultOnSet(
                new SetRequestMap().withBids(
                        new BidSetItemMap().withKeywordId(keywordId).withStrategyPriority(priority)
                ),
                BidExpectedResult.successWithKeywordId()
        );

        GetResponse getResponse = api.userSteps.bidsSteps().bidsGetByKeywordIdsWithFields(
                Arrays.asList(BidFieldEnum.KEYWORD_ID, BidFieldEnum.STRATEGY_PRIORITY),
                keywordId
        );
        assertThat("получили правильное значение StrategyPriority",
                getResponse.getBids(),
                beanDifferV5(
                        Arrays.asList(
                                new BidsGetItemMap().withKeywordId(keywordId).withStrategyPriority(
                                        new ObjectFactory().createBidGetItemStrategyPriority(priority)
                                ).getBean()
                        )
                )
        );
    }

    @Test
    @TestCaseId("2732")
    public void setStrategyPriorityForAutotargeting() {
        Long autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);
        api.userSteps.bidsSteps().shouldGetBidResultOnSet(
                new SetRequestMap().withBids(
                        new BidSetItemMap().withKeywordId(autotargetingId).withStrategyPriority(priority)
                ),
                BidExpectedResult.successWithKeywordId()
        );

        GetResponse getResponse = api.userSteps.bidsSteps().bidsGetByKeywordIdsWithFields(
                Arrays.asList(BidFieldEnum.KEYWORD_ID, BidFieldEnum.STRATEGY_PRIORITY),
                autotargetingId
        );
        assertThat("получили правильное значение StrategyPriority",
                getResponse.getBids(),
                beanDifferV5(
                        Arrays.asList(
                                new BidsGetItemMap().withKeywordId(autotargetingId).withStrategyPriority(
                                        new ObjectFactory().createBidGetItemStrategyPriority(priority)
                                ).getBean()
                        )
                )
        );
    }

    @Test
    @TestCaseId("2733")
    public void setStrategyPriorityForKeywordAndAutotargeting() {
        Long keywordId = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
        Long autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);
        api.userSteps.bidsSteps().shouldGetBidResultOnSet(
                new SetRequestMap()
                        .withBids(
                                new BidSetItemMap()
                                        .withKeywordId(keywordId)
                                        .withStrategyPriority(priority),
                                new BidSetItemMap()
                                        .withKeywordId(autotargetingId)
                                        .withStrategyPriority(priority)
                        ),
                BidExpectedResult.successWithKeywordId(),
                BidExpectedResult.successWithKeywordId()
        );

        GetResponse getResponse = api.userSteps.bidsSteps().bidsGet(
                new GetRequestMap().withSelectionCriteria(
                        new BidsSelectionCriteriaMap().withKeywordIds(keywordId, autotargetingId)
                ).withFieldNames(BidFieldEnum.KEYWORD_ID, BidFieldEnum.STRATEGY_PRIORITY)
        );
        assertThat("получили правильные значения StrategyPriority",
                getResponse.getBids(),
                beanDifferV5(
                        Arrays.asList(
                                new BidsGetItemMap().withKeywordId(keywordId).withStrategyPriority(
                                        new ObjectFactory().createBidGetItemStrategyPriority(priority)
                                ).getBean(),
                                new BidsGetItemMap().withKeywordId(autotargetingId).withStrategyPriority(
                                        new ObjectFactory().createBidGetItemStrategyPriority(priority)
                                ).getBean()
                        )
                )
        );
    }

}
