package ru.yandex.autotests.direct.api.bids.setauto;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import com.yandex.direct.api.v5.bids.BidFieldEnum;
import com.yandex.direct.api.v5.bids.BidGetItem;
import com.yandex.direct.api.v5.bids.CalculateByEnum;
import com.yandex.direct.api.v5.general.PositionEnum;
import com.yandex.direct.api.v5.general.PriorityEnum;
import com.yandex.direct.api.v5.general.ScopeEnum;
import org.apache.commons.lang.StringUtils;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bids.BidsFeatures;
import ru.yandex.autotests.direct.api.bids.BidsLogins;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.bids.BidExpectedResult;
import ru.yandex.autotests.directapi.model.api5.bids.BidSetAutoItemMap;
import ru.yandex.autotests.directapi.model.api5.bids.BidSetItemMap;
import ru.yandex.autotests.directapi.model.api5.bids.SetAutoRequestMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.MobileAppCampaignNetworkStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.MobileAppCampaignSearchStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignNetworkStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignSearchStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.TestCaseId;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by gerdler on 25.04.17.
 */
@Aqua.Test
@Features(BidsFeatures.SET_AUTO)
@RunWith(Parameterized.class)
public class SetAutoAutotargetingTest {

    private static final String CLIENT = BidsLogins.BIDS_CLIENT;
    private static final Long MAX_BID = MoneyCurrency.get(Currency.RUB).getLongMaxPrice().longValue();
    private static final Long MAX_SET_AUTO_BID = MAX_BID / 2L;

    private static Long adGroupId;
    private static Long autotargetingId;
    private static Long adGroupId2;
    private static Long autotargetingId2;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public Runnable adGroupCreator;

    @Parameterized.Parameters(name = "{0}")
    public static Collection testData() {
        Long textCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaignWithStrategies(
                new TextCampaignSearchStrategyAddMap().defaultHighestPosition(),
                new TextCampaignNetworkStrategyAddMap().defaultMaximumCoverage()
        );
        Long mobileAppCampaignId = api.userSteps.campaignSteps().addDefaultMobileAppCampaignWithStrategies(
                new MobileAppCampaignSearchStrategyAddMap().defaultHighestPosition(),
                new MobileAppCampaignNetworkStrategyAddMap().defaultMaximumCoverage()
        );

        Object[][] data = new Object[][]{
                {
                        "text campaign",
                        (Runnable)() -> {
                            adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(textCampaignId);
                            api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
                            adGroupId2 = api.userSteps.adGroupsSteps().addDefaultGroup(textCampaignId);
                            api.userSteps.adsSteps().addDefaultTextAd(adGroupId2);
                        }
                },
                {
                        "mobile app campaign",
                        (Runnable)() -> {
                            adGroupId = api.userSteps.adGroupsSteps().addDefaultMobileGroup(mobileAppCampaignId);
                            api.userSteps.adsSteps().addDefaultMobileAppAd(adGroupId);
                            adGroupId2 = api.userSteps.adGroupsSteps().addDefaultMobileGroup(mobileAppCampaignId);
                            api.userSteps.adsSteps().addDefaultMobileAppAd(adGroupId2);
                        }
                }
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    @Step("Подготовка данных для теста")
    public static void clearUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
    }

    @Before
    public void prepareData() {
        adGroupCreator.run();

        autotargetingId = api.userSteps.keywordsSteps().addAutotargetingWithBidsAndStrategyPriority(adGroupId, MAX_BID, MAX_BID, PriorityEnum.HIGH);

        api.userSteps.keywordsSteps().addKeyword(adGroupId2, "kw1");
        api.userSteps.keywordsSteps().addKeyword(adGroupId2, "kw2");
        api.userSteps.keywordsSteps().addKeyword(adGroupId2, "kw3");
        autotargetingId2 = api.userSteps.keywordsSteps().addAutotargetingWithBidsAndStrategyPriority(adGroupId2, MAX_BID, MAX_BID, PriorityEnum.HIGH);
    }

    @Test
    @TestCaseId("2743")
    public void setAutoByAutotargetingId() {
        api.userSteps.bidsSteps().shouldGetBidResultOnSetAuto(
                new SetAutoRequestMap().withBids(
                        new BidSetAutoItemMap()
                                .withKeywordId(autotargetingId)
                                .withPosition(PositionEnum.FOOTERFIRST)
                                .withIncreasePercent(50)
                                .withCalculateBy(CalculateByEnum.VALUE)
                                .withContextCoverage(3)
                                .withMaxBid(MAX_SET_AUTO_BID)
                                .withScope(ScopeEnum.SEARCH, ScopeEnum.NETWORK)
                ),
                BidExpectedResult.errors(
                        new Notification(6001,
                                Api5ErrorDetails.SET_AUTO_DENIED_FOR_AUTOTARGETING,
                                StringUtils.capitalize(BidSetItemMap.KEYWORD_ID),
                                autotargetingId
                        )
                )
        );
    }

    @Test
    @TestCaseId("2744")
    public void setAutoOnSearchByAdGroupIdInAdGroupWithoutKeywords() {
            api.userSteps.bidsSteps().shouldGetBidResultOnSetAuto(
                    new SetAutoRequestMap().withBids(
                            new BidSetAutoItemMap()
                                    .withAdGroupId(adGroupId)
                                    .withPosition(PositionEnum.FOOTERFIRST)
                                    .withIncreasePercent(50)
                                    .withCalculateBy(CalculateByEnum.VALUE)
                                    .withMaxBid(MAX_SET_AUTO_BID)
                                    .withScope(ScopeEnum.SEARCH)
                    ),
                    BidExpectedResult.successWithAdGroupId()
            );

            List<BidGetItem> bidsItemList = api.userSteps.bidsSteps().bidsGetByKeywordIdsWithFields(Collections.singletonList(BidFieldEnum.BID), autotargetingId).getBids();
            assumeThat("получили ставки автотаргетинга", bidsItemList.size(), equalTo(1));
            assertThat("ставка на поиске у автотаргетинга не изменилась", bidsItemList.get(0).getBid(), equalTo(MAX_BID));
    }

    @Test
    @TestCaseId("2751")
    public void setAutoOnSearchByAdGroupIdInAdGroupWithKeywords() {
        api.userSteps.bidsSteps().shouldGetBidResultOnSetAuto(
                new SetAutoRequestMap().withBids(
                        new BidSetAutoItemMap()
                                .withAdGroupId(adGroupId2)
                                .withPosition(PositionEnum.FOOTERFIRST)
                                .withIncreasePercent(50)
                                .withCalculateBy(CalculateByEnum.VALUE)
                                .withMaxBid(MAX_SET_AUTO_BID)
                                .withScope(ScopeEnum.SEARCH)
                ),
                BidExpectedResult.successWithAdGroupId()
        );

        List<BidGetItem> bidsItemList = api.userSteps.bidsSteps()
                .bidsGetByKeywordIdsWithFields(Collections.singletonList(BidFieldEnum.BID), autotargetingId2).getBids();
        assumeThat("получили ставку автотаргетинга", bidsItemList.size(), equalTo(1));
        assertThat("ставка на поиске у автотаргетинга изменилась", bidsItemList.get(0).getBid(), not(equalTo(MAX_BID)));
    }

    @Test
    @TestCaseId("2752")
    public void setAutoOnNetworkByAdGroupIdInAdGroupWithoutKeywords() {
        api.userSteps.bidsSteps().shouldGetBidResultOnSetAuto(
                new SetAutoRequestMap().withBids(
                        new BidSetAutoItemMap()
                                .withAdGroupId(adGroupId)
                                .withMaxBid(MAX_SET_AUTO_BID)
                                .withContextCoverage(100)
                                .withScope(ScopeEnum.NETWORK)
                ),
                BidExpectedResult.successWithAdGroupId()
        );

        List<BidGetItem> bidsItemList = api.userSteps.bidsSteps().bidsGetByKeywordIdsWithFields(Collections.singletonList(BidFieldEnum.BID), autotargetingId).getBids();
        assumeThat("получили ставку автотаргетинга", bidsItemList.size(), equalTo(1));
        assumeThat("ставка на поиске у автотаргетинга не изменилась", bidsItemList.get(0).getBid(), equalTo(MAX_BID));
    }

    @Test
    @TestCaseId("2753")
    public void setAutoOnNetworkByAdGroupIdInAdGroupWithKeywords() {
        api.userSteps.bidsSteps().shouldGetBidResultOnSetAuto(
                new SetAutoRequestMap().withBids(
                        new BidSetAutoItemMap()
                                .withAdGroupId(adGroupId2)
                                .withMaxBid(MAX_SET_AUTO_BID)
                                .withContextCoverage(100)
                                .withScope(ScopeEnum.NETWORK)
                ),
                BidExpectedResult.successWithAdGroupId()
        );

        List<BidGetItem> bidsItemList = api.userSteps.bidsSteps().bidsGetByKeywordIdsWithFields(Collections.singletonList(BidFieldEnum.BID), autotargetingId2).getBids();
        assumeThat("получили ставку автотаргетинга", bidsItemList.size(), equalTo(1));
        assumeThat("ставка на поиске у автотаргетинга не изменилась", bidsItemList.get(0).getBid(), equalTo(MAX_BID));
    }
}
