package ru.yandex.autotests.direct.api.bids.setauto;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.bids.AuctionBidItem;
import com.yandex.direct.api.v5.bids.BidFieldEnum;
import com.yandex.direct.api.v5.bids.BidGetItem;
import com.yandex.direct.api.v5.bids.CalculateByEnum;
import com.yandex.direct.api.v5.bids.GetResponse;
import com.yandex.direct.api.v5.bids.SearchPrices;
import com.yandex.direct.api.v5.general.PositionEnum;
import com.yandex.direct.api.v5.general.ScopeEnum;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bids.BidsFeatures;
import ru.yandex.autotests.direct.api.bids.BidsLogins;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.bids.BidExpectedResult;
import ru.yandex.autotests.directapi.model.api5.bids.BidSetAutoItemMap;
import ru.yandex.autotests.directapi.model.api5.bids.BidsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.bids.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.bids.SetAutoRequestMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignNetworkStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignSearchStrategyAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.having;
import static ch.lambdaj.Lambda.on;
import static ch.lambdaj.Lambda.selectFirst;
import static org.hamcrest.Matchers.comparesEqualTo;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by ginger on 01.10.14.
 */

@Aqua.Test
@Features(BidsFeatures.SET_AUTO)
@RunWith(Parameterized.class)
@Issue("DIRECT-35999")
public class SetAutoFormulaForComputationBidTest {
    private static LogSteps log = LogSteps.getLogger(SetAutoFormulaForComputationBidTest.class);
    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();
    private static long keywordId;

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final int INCREASE_PERCENT = 100;

    @Parameterized.Parameter(value = 0)
    public PositionEnum position;

    @Parameterized.Parameter(value = 1)
    public CalculateByEnum calculateBy;

    @Parameterized.Parameter(value = 2)
    public PriceCalculator priceCalculator;

    @Parameterized.Parameters(name = "position = {0}, calculateBy = {1}")
    public static Collection data() {
        final BigDecimal INCREASE_PERCENT_DIVIDE_BY_100 = new BigDecimal(INCREASE_PERCENT).divide(new BigDecimal(100));
        return Arrays.asList(new Object[][]{
                {PositionEnum.FOOTERBLOCK, CalculateByEnum.VALUE, new PriceCalculator() {
                    @Override
                    public BigDecimal calculatePrice(BidGetItem bidGetItem) {
                        SearchPrices minSearchPrice =
                                selectFirst(bidGetItem.getSearchPrices(),
                                        having(on(SearchPrices.class).getPosition(),
                                                equalTo(PositionEnum.FOOTERBLOCK)));
                        BigDecimal min = getBigDecimal(minSearchPrice.getPrice());
                        return min.add(min.multiply(INCREASE_PERCENT_DIVIDE_BY_100));
                    }
                }},
                {PositionEnum.FOOTERBLOCK, CalculateByEnum.DIFF, new PriceCalculator() {
                    @Override
                    public BigDecimal calculatePrice(BidGetItem bidGetItem) {
                        SearchPrices minSearchPrice =
                                selectFirst(bidGetItem.getSearchPrices(),
                                        having(on(SearchPrices.class).getPosition(),
                                                equalTo(PositionEnum.FOOTERBLOCK)));
                        SearchPrices maxSearchPrice =
                                selectFirst(bidGetItem.getSearchPrices(),
                                        having(on(SearchPrices.class).getPosition(),
                                                equalTo(PositionEnum.FOOTERFIRST)));
                        BigDecimal min = getBigDecimal(minSearchPrice.getPrice());
                        BigDecimal max = getBigDecimal(maxSearchPrice.getPrice());
                        return min.add(max.subtract(min).multiply(INCREASE_PERCENT_DIVIDE_BY_100));
                    }
                }},
                {PositionEnum.FOOTERFIRST, CalculateByEnum.VALUE, new PriceCalculator() {
                    @Override
                    public BigDecimal calculatePrice(BidGetItem bidGetItem) {
                        SearchPrices maxSearchPrice =
                                selectFirst(bidGetItem.getSearchPrices(),
                                        having(on(SearchPrices.class).getPosition(),
                                                equalTo(PositionEnum.FOOTERFIRST)));
                        BigDecimal max = getBigDecimal(maxSearchPrice.getPrice());
                        return max.add(max.multiply(INCREASE_PERCENT_DIVIDE_BY_100));
                    }
                }},
                {PositionEnum.FOOTERFIRST, CalculateByEnum.DIFF, new PriceCalculator() {
                    @Override
                    public BigDecimal calculatePrice(BidGetItem bidGetItem) {
                        SearchPrices maxSearchPrice =
                                selectFirst(bidGetItem.getSearchPrices(),
                                        having(on(SearchPrices.class).getPosition(),
                                                equalTo(PositionEnum.FOOTERFIRST)));
                        SearchPrices premiumMinSearchPrice =
                                selectFirst(bidGetItem.getSearchPrices(),
                                        having(on(SearchPrices.class).getPosition(),
                                                equalTo(PositionEnum.PREMIUMBLOCK)));
                        BigDecimal premiumMin = getBigDecimal(premiumMinSearchPrice.getPrice());
                        BigDecimal max = getBigDecimal(maxSearchPrice.getPrice());
                        BigDecimal diff = premiumMin.subtract(max);
                        if (diff.compareTo(BigDecimal.ZERO) < 0) {
                            diff = BigDecimal.ZERO;
                        }
                        return max.add(diff.multiply(INCREASE_PERCENT_DIVIDE_BY_100));
                    }
                }},
                {PositionEnum.PREMIUMBLOCK, CalculateByEnum.VALUE, new PriceCalculator() {
                    @Override
                    public BigDecimal calculatePrice(BidGetItem bidGetItem) {
                        SearchPrices premiumMinSearchPrice =
                                selectFirst(bidGetItem.getSearchPrices(),
                                        having(on(SearchPrices.class).getPosition(),
                                                equalTo(PositionEnum.PREMIUMBLOCK)));
                        BigDecimal premiumMin = getBigDecimal(premiumMinSearchPrice.getPrice());
                        return premiumMin.add(premiumMin.multiply(INCREASE_PERCENT_DIVIDE_BY_100));
                    }
                }},
                {PositionEnum.PREMIUMBLOCK, CalculateByEnum.DIFF, new PriceCalculator() {
                    @Override
                    public BigDecimal calculatePrice(BidGetItem bidGetItem) {
                        AuctionBidItem auctionBidPremiumMax2 = bidGetItem.getAuctionBids().stream()
                                .filter(auctionBid -> auctionBid.getPosition().equals(PositionEnum.P_13.value()))
                                .findAny().orElseThrow(() -> new AssertionError("в AuctionBids нет требуемой позиции"));
                        SearchPrices premiumMinSearchPrice =
                                selectFirst(bidGetItem.getSearchPrices(),
                                        having(on(SearchPrices.class).getPosition(),
                                                equalTo(PositionEnum.PREMIUMBLOCK)));
                        BigDecimal premiumMin = getBigDecimal(premiumMinSearchPrice.getPrice());
                        BigDecimal premiumMax2 = getBigDecimal(auctionBidPremiumMax2.getBid());
                        BigDecimal diff = premiumMax2.subtract(premiumMin);
                        if (diff.compareTo(BigDecimal.ZERO) < 0) {
                            diff = BigDecimal.ZERO;
                        }
                        return premiumMin.add(diff.multiply(INCREASE_PERCENT_DIVIDE_BY_100));
                    }
                }},
                {PositionEnum.PREMIUMFIRST, CalculateByEnum.VALUE, new PriceCalculator() {
                    @Override
                    public BigDecimal calculatePrice(BidGetItem bidGetItem) {
                        SearchPrices premiumMaxSearchPrice =
                                selectFirst(bidGetItem.getSearchPrices(),
                                        having(on(SearchPrices.class).getPosition(),
                                                equalTo(PositionEnum.PREMIUMFIRST)));
                        BigDecimal premiumMax = getBigDecimal(premiumMaxSearchPrice.getPrice());
                        return premiumMax.add(premiumMax.multiply(INCREASE_PERCENT_DIVIDE_BY_100));
                    }
                }},
                {PositionEnum.PREMIUMFIRST, CalculateByEnum.DIFF, new PriceCalculator() {
                    @Override
                    public BigDecimal calculatePrice(BidGetItem bidGetItem) {
                        SearchPrices premiumMaxSearchPrice =
                                selectFirst(bidGetItem.getSearchPrices(),
                                        having(on(SearchPrices.class).getPosition(),
                                                equalTo(PositionEnum.PREMIUMFIRST)));
                        BigDecimal premiumMax = getBigDecimal(premiumMaxSearchPrice.getPrice());
                        return premiumMax;
                    }
                }}
        });
    }

    @BeforeClass
    @Step("Подготовка данных для теста")
    public static void readBanner() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(BidsLogins.BIDS_CLIENT);
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaignWithStrategies(
                new TextCampaignSearchStrategyAddMap().defaultHighestPosition(),
                new TextCampaignNetworkStrategyAddMap().defaultMaximumCoverage(),
                BidsLogins.BIDS_CLIENT
        );
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId, BidsLogins.BIDS_CLIENT);
        api.userSteps.adsSteps().addDefaultTextAd(adGroupId, BidsLogins.BIDS_CLIENT);
        keywordId = api.userSteps.keywordsSteps().addKeyword(BidsLogins.BIDS_CLIENT, adGroupId, "арендовать комнату");
        api.protocol(ProtocolType.SOAP);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("1497")
    public void checkPriceValueInBidsGet() {
        BigDecimal expectedPrice = getExpectedPrice(priceCalculator);
        api.userSteps.bidsSteps().shouldGetBidResultOnSetAuto(
                new SetAutoRequestMap()
                        .withBids(
                                new BidSetAutoItemMap()
                                        .withKeywordId(keywordId)
                                        .withIncreasePercent(INCREASE_PERCENT)
                                        .withCalculateBy(calculateBy)
                                        .withPosition(position)
                                        .withMaxBid(MoneyCurrency.get(Currency.RUB).getLongMaxPrice().longValue())
                                        .withScope(ScopeEnum.SEARCH)
                        ),
                BidsLogins.BIDS_CLIENT,
                BidExpectedResult.successWithKeywordId());
        GetResponse getResponse = api.userSteps.bidsSteps().bidsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new BidsSelectionCriteriaMap().withKeywordIds(keywordId)
                        )
                        .withFieldNames(BidFieldEnum.BID),
                BidsLogins.BIDS_CLIENT
        );
        BigDecimal actualPrice = getBigDecimal(getResponse.getBids().get(0).getBid());
        log.info("Ожидаемое значение " + expectedPrice);
        log.info("Полученное значение " + actualPrice);
        assertThat("значение поля Bid соответствует ожидаемому",
                actualPrice,
                comparesEqualTo(expectedPrice));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("1498")
    public void checkMaxPriceInGetBannerPhrase() {
        api.userSteps.bidsSteps().shouldGetBidResultOnSetAuto(
                new SetAutoRequestMap()
                        .withBids(
                                new BidSetAutoItemMap()
                                        .withKeywordId(keywordId)
                                        .withIncreasePercent(INCREASE_PERCENT)
                                        .withCalculateBy(calculateBy)
                                        .withPosition(position)
                                        .withMaxBid(300000L)
                                        .withScope(ScopeEnum.SEARCH)
                        ),
                BidsLogins.BIDS_CLIENT,
                BidExpectedResult.successWithKeywordId());

        BidGetItem bid = api.userSteps.bidsSteps().bidsGet(
                new GetRequestMap().withSelectionCriteria(
                                new BidsSelectionCriteriaMap().withKeywordIds(keywordId)).withFieldNames(BidFieldEnum.BID),
                BidsLogins.BIDS_CLIENT).getBids().get(0);
        assertThat("значение поля Bid соответствует ожидаемому", bid.getBid(), equalTo(300000L));
    }

    private static interface PriceCalculator {
        public BigDecimal calculatePrice(BidGetItem bidGetItem);
    }

    private static BigDecimal getBigDecimal(Long value) {
        return BigDecimal.valueOf(value);
    }

    private static BigDecimal getExpectedPrice(PriceCalculator priceCalculator) {
        GetResponse getResponse = api.userSteps.bidsSteps().bidsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new BidsSelectionCriteriaMap().withKeywordIds(keywordId)
                        )
                        .withFieldNames(BidFieldEnum.BID, BidFieldEnum.SEARCH_PRICES, BidFieldEnum.AUCTION_BIDS),
                BidsLogins.BIDS_CLIENT
        );
        assumeThat("получили верное кол-во ставок в get", getResponse.getBids().size(), equalTo(1));
        return priceCalculator.calculatePrice(getResponse.getBids().get(0));
    }

}
