package ru.yandex.autotests.direct.api.keywordbids.set;

import java.util.Arrays;
import java.util.Collection;
import ru.yandex.qatools.allure.annotations.TestCaseId;

import com.yandex.direct.api.v5.general.PriorityEnum;
import com.yandex.direct.api.v5.keywordbids.GetResponse;
import com.yandex.direct.api.v5.keywordbids.KeywordBidFieldEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.bids.BidsLogins;
import ru.yandex.autotests.direct.api.keywordbids.KeywordBidsFeatures;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignNetworkStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignSearchStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.keywordbids.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywordbids.KeywordBidExpectedResult;
import ru.yandex.autotests.directapi.model.api5.keywordbids.KeywordBidsGetItemMap;
import ru.yandex.autotests.directapi.model.api5.keywordbids.KeywordBidsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.keywordbids.KeywordBidsSetItemMap;
import ru.yandex.autotests.directapi.model.api5.keywordbids.SetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by gerdler on 11.04.17.
 */
@Aqua.Test
@Features(KeywordBidsFeatures.SET)
@RunWith(Parameterized.class)
public class SetStrategyPriorityPositiveTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(BidsLogins.BIDS_CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    @Parameterized.Parameter(0)
    public PriorityEnum priority;

    @Parameterized.Parameters(name = "StrategyPriority = {0}")
    public static Collection testData() {
        Object[][] data = new Object[][]{
                {PriorityEnum.HIGH},
                {PriorityEnum.NORMAL},
                {PriorityEnum.LOW},
        };
        return Arrays.asList(data);
    }

    private static Long campaignId;
    private Long adGroupId;

    @BeforeClass
    @Step("Подготовка данных для теста")
    public static void createCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(BidsLogins.BIDS_CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaignWithStrategies(
                new TextCampaignSearchStrategyAddMap().defaultWbMaximumClicks(Currency.RUB),
                new TextCampaignNetworkStrategyAddMap().defaultNetworkDefault()
        );
    }

    @Before
    public void createAdGroup() {
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
    }

    @Test
    @TestCaseId("2921")
    public void setStrategyPriorityForKeyword() {
        Long keywordId = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
        api.userSteps.keywordBidsSteps().shouldGetKeywordBidResultOnSet(
                new SetRequestMap().withKeywordBids(
                        new KeywordBidsSetItemMap().withKeywordId(keywordId).withStrategyPriority(priority)
                ),
                KeywordBidExpectedResult.successWithKeywordId()
        );

        GetResponse getResponse = api.userSteps.keywordBidsSteps().bidsGetByKeywordIdsWithFields(
                Arrays.asList(KeywordBidFieldEnum.KEYWORD_ID, KeywordBidFieldEnum.STRATEGY_PRIORITY),
                keywordId
        );
        assertThat("получили правильное значение StrategyPriority",
                getResponse.getKeywordBids(),
                beanDifferV5(
                        Arrays.asList(
                                new KeywordBidsGetItemMap().withKeywordId(keywordId).withStrategyPriority(priority
                                ).getBean()
                        )
                )
        );
    }

    @Test
    @TestCaseId("2920")
    public void setStrategyPriorityForAutotargeting() {
        Long autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);
        api.userSteps.keywordBidsSteps().shouldGetKeywordBidResultOnSet(
                new SetRequestMap().withKeywordBids(
                        new KeywordBidsSetItemMap().withKeywordId(autotargetingId).withStrategyPriority(priority)
                ),
                KeywordBidExpectedResult.successWithKeywordId()
        );

        GetResponse getResponse = api.userSteps.keywordBidsSteps().bidsGetByKeywordIdsWithFields(
                Arrays.asList(KeywordBidFieldEnum.KEYWORD_ID, KeywordBidFieldEnum.STRATEGY_PRIORITY),
                autotargetingId
        );
        assertThat("получили правильное значение StrategyPriority",
                getResponse.getKeywordBids(),
                beanDifferV5(
                        Arrays.asList(
                                new KeywordBidsGetItemMap().withKeywordId(autotargetingId).withStrategyPriority(
                                        priority
                                ).getBean()
                        )
                )
        );
    }

    @Test
    @TestCaseId("2922")
    public void setStrategyPriorityForKeywordAndAutotargeting() {
        Long keywordId = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupId);
        Long autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);
        api.userSteps.keywordBidsSteps().shouldGetKeywordBidResultOnSet(
                new SetRequestMap()
                        .withKeywordBids(
                                new KeywordBidsSetItemMap()
                                        .withKeywordId(keywordId)
                                        .withStrategyPriority(priority),
                                new KeywordBidsSetItemMap()
                                        .withKeywordId(autotargetingId)
                                        .withStrategyPriority(priority)
                        ),
                KeywordBidExpectedResult.successWithKeywordId(),
                KeywordBidExpectedResult.successWithKeywordId()
        );

        GetResponse getResponse = api.userSteps.keywordBidsSteps().keywordBidsGet(
                new GetRequestMap().withSelectionCriteria(
                        new KeywordBidsSelectionCriteriaMap().withKeywordIds(keywordId, autotargetingId)
                ).withFieldNames(KeywordBidFieldEnum.KEYWORD_ID, KeywordBidFieldEnum.STRATEGY_PRIORITY)
        );
        assertThat("получили правильные значения StrategyPriority",
                getResponse.getKeywordBids(),
                beanDifferV5(
                        Arrays.asList(
                                new KeywordBidsGetItemMap().withKeywordId(keywordId).withStrategyPriority(
                                        priority
                                ).getBean(),
                                new KeywordBidsGetItemMap().withKeywordId(autotargetingId).withStrategyPriority(
                                        priority
                                ).getBean()
                        )
                )
        );
    }

}
