package ru.yandex.autotests.direct.api.campaigns;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.hamcrest.Matcher;
import org.hamcrest.core.AnyOf;
import uk.co.it.modular.hamcrest.date.DateMatchers;

import ru.yandex.autotests.direct.db.beans.campaign.StrategyData;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampOptionsStrategy;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.StrategiesArchived;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.StrategiesAttributionModel;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.StrategiesDayBudgetShowMode;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.StrategiesEnableCpcHold;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.StrategiesIsPublic;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.StrategiesType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.StrategyMetrikaCountersSource;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.Strategies;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampOptionsRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampaignsRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.MetrikaCountersRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.StrategiesRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.StrategyMetrikaCountersRecord;
import ru.yandex.autotests.irt.testutils.beandiffer2.beanfield.BeanFieldPath;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategies;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategy;

import static org.hamcrest.CoreMatchers.anyOf;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.Matchers.anything;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.beanfield.BeanFieldPath.newPath;

@ParametersAreNonnullByDefault
public class CampaignCommonMatchers {

    public static AnyOf<String> getMeaningfulGoalsJsonMatcher(Long meaningfulGoalId, Long meaningfulGoalValue) {
        return anyOf(
                equalTo("[{\"goal_id\": " + meaningfulGoalId + ", \"value\": \"" + meaningfulGoalValue + "\"}]"),
                equalTo("[{\"value\": \"" + meaningfulGoalValue + "\", \"goal_id\": " + meaningfulGoalId + "}]"),
                // через интерфейс фронт автоматом форматирует до двух знаков после запятой
                // поэтому это норм, что и api5 будет делать так же (менять 1, на 1.00)
                equalTo("[{\"goal_id\": " + meaningfulGoalId + ", \"value\": \"" + meaningfulGoalValue + ".00\"}]"),
                equalTo("[{\"value\": \"" + meaningfulGoalValue + ".00\", \"goal_id\": " + meaningfulGoalId + "}]")
        );
    }

    public static void compareCampaignAndStrategy(
            CampaignsRecord campaignsRecord,
            CampOptionsRecord campOptionsRecord,
            List<MetrikaCountersRecord> metrikaCountersRecordList,
            StrategiesRecord strategiesRecord,
            List<StrategyMetrikaCountersRecord> strategyMetrikaCountersList,
            Matcher<String> meaningfulGoalsMatcher,
            Boolean strategyIsPublic,
            @Nullable String name) {
        StrategiesRecord expectedStrategiesRecord = new StrategiesRecord()
                .setStrategyId(campaignsRecord.getStrategyId())
                .setClientid(campaignsRecord.getClientid())
                .setType(StrategiesType.valueOf(campaignsRecord.getStrategyName().name()))
                .setStrategyData(campaignsRecord.getStrategyData().toString())
                .setContextlimit(campaignsRecord.getContextlimit())
                .setAttributionModel(StrategiesAttributionModel.valueOf(campaignsRecord.getAttributionModel().name()))
                .setDayBudget(campaignsRecord.getDayBudget())
                .setDayBudgetShowMode(StrategiesDayBudgetShowMode.valueOf(campaignsRecord.getDayBudgetShowMode().name()))
                .setWalletCid(campaignsRecord.getWalletCid())
                .setArchived(StrategiesArchived.No)
                .setDayBudgetDailyChangeCount(campOptionsRecord.getDayBudgetDailyChangeCount())
                .setEnableCpcHold(campaignsRecord.getOpts().contains("enable_cpc_hold")
                        ? StrategiesEnableCpcHold.Yes
                        : StrategiesEnableCpcHold.No)
                .setMeaningfulGoals(campOptionsRecord.getMeaningfulGoals())
                .setIsPublic(strategyIsPublic ? StrategiesIsPublic.Yes : StrategiesIsPublic.No)
                .setName(name);
        StrategyData strategiesStrategyData = StrategyData.fromString(strategiesRecord.getStrategyData());
        StrategyData campaignsStrategyData = StrategyData.fromString(campaignsRecord.getStrategyData().toString());

        assertThat("стратегия правильно сохранилась в базе",
                strategiesRecord.intoMap(), beanDifferV5(expectedStrategiesRecord.intoMap())
                        .useCompareStrategy(DefaultCompareStrategies.allFieldsExcept(
                                BeanFieldPath.newPath(Strategies.STRATEGIES.STRATEGY_DATA.getName()),
                                BeanFieldPath.newPath(Strategies.STRATEGIES.DAY_BUDGET_LAST_CHANGE.getName())
                        ).forFields(BeanFieldPath.newPath(Strategies.STRATEGIES.LASTCHANGE.getName()))
                                .useMatcher(DateMatchers.within(1, TimeUnit.SECONDS, campaignsRecord.getLastchange()))
                                .forFields(
                                        BeanFieldPath.newPath(Strategies.STRATEGIES.MEANINGFUL_GOALS.getName()))
                                .useMatcher(meaningfulGoalsMatcher)));
        if (campOptionsRecord.getDayBudgetLastChange() == null) {
            assertThat("стратегия правильно сохранилась в базе",
                    strategiesRecord.getDayBudgetLastChange(), nullValue());
        } else {
            assertThat("стратегия правильно сохранилась в базе",
                    strategiesRecord.getDayBudgetLastChange(),
                    DateMatchers.within(1, TimeUnit.SECONDS, campOptionsRecord.getDayBudgetLastChange()));
        }
        assertThat("стратегия правильно сохранилась в базе",
                strategiesStrategyData,
                beanDifferV5(campaignsStrategyData).useCompareStrategy(DefaultCompareStrategies.allFieldsExcept(
                        BeanFieldPath.newPath("version")
                )));

        List<Map<String, Object>> expectedStrategyMetrikaCounters = new ArrayList<>();
        for (MetrikaCountersRecord metrikaCountersRecord : metrikaCountersRecordList) {
            StrategyMetrikaCountersRecord record = new StrategyMetrikaCountersRecord()
                    .setStrategyId(strategiesRecord.getStrategyId())
                    .setMetrikaCounter(metrikaCountersRecord.getMetrikaCounter())
                    .setSource(StrategyMetrikaCountersSource.valueOf(metrikaCountersRecord.getSource().name()))
                    .setIsDeleted(metrikaCountersRecord.getIsDeleted())
                    .setHasEcommerce(metrikaCountersRecord.getHasEcommerce());
            expectedStrategyMetrikaCounters.add(record.intoMap());
        }

        List<Map<String, Object>> actualStrategyMetrikaCounters = new ArrayList<>();
        for (StrategyMetrikaCountersRecord strategyMetrikaCountersRecord : strategyMetrikaCountersList) {
            actualStrategyMetrikaCounters.add(strategyMetrikaCountersRecord.intoMap());
        }

        assertThat("счетчики метрики стратегии правильно сохранились в базе",
                actualStrategyMetrikaCounters, beanDifferV5(expectedStrategyMetrikaCounters));
    }

    public static void compareMetrikaCountersBetweenTwoStrategies(
            List<StrategyMetrikaCountersRecord> actual,
            List<StrategyMetrikaCountersRecord> expected
    ) {
        List<Map<String, Object>> actualList = new ArrayList<>();
        for (StrategyMetrikaCountersRecord strategyMetrikaCountersRecord : actual) {
            actualList.add(strategyMetrikaCountersRecord.intoMap());
        }

        List<Map<String, Object>> expectedList = new ArrayList<>();
        for (StrategyMetrikaCountersRecord strategyMetrikaCountersRecord : expected) {
            expectedList.add(strategyMetrikaCountersRecord.intoMap());
        }

        assertThat("счетчики метрики стратегии правильно сохранились в базе",
                actualList, beanDifferV5(expectedList).useCompareStrategy(DefaultCompareStrategies.allFieldsExcept(
                        BeanFieldPath.newPath(".+", "has_ecommerce")
                )));
    }

    public static DefaultCompareStrategy strategyDataCompareStrategy =
            DefaultCompareStrategies.onlyExpectedFields()
                    .forFields(newPath("strategy_data/last_bidder_restart_time")).useMatcher(anything())
                    .forFields(newPath("strategy_data/goal_id"))
                    .useDiffer(new LongOrStringDiffer()); // perl заполняет строкой, java - числом

}
