package ru.yandex.autotests.direct.api.campaigns.add;

import java.math.BigDecimal;
import java.math.RoundingMode;

import com.yandex.direct.api.v5.campaigns.TextCampaignSettingsEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.db.beans.campaign.StrategyData;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStrategyName;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.StrategiesArchived;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.StrategiesAttributionModel;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.StrategiesDayBudgetShowMode;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.StrategiesEnableCpcHold;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.StrategiesIsPublic;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.StrategiesType;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.StrategyMetrikaCountersSource;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.Strategies;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampOptionsRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampaignsRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.StrategiesRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.StrategyMetrikaCountersRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppcdict.tables.records.ShardIncStrategyIdRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.PriorityGoalsArrayMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.PriorityGoalsItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignNetworkStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignSearchStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignSettingMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignStrategyAddMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.beandiffer2.beanfield.BeanFieldPath;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategies;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.CoreMatchers.not;
import static ru.yandex.autotests.direct.api.campaigns.CampaignCommonMatchers.getMeaningfulGoalsJsonMatcher;
import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static uk.co.it.modular.hamcrest.date.DateMatchers.before;

@Aqua.Test
@Features(CampaignsFeatures.ADD)
@Description("Проверка сохранения в базу пакетных стратегий при создании кампании")
@Issue("https://st.yandex-team.ru/DIRECT-160676")
public class AddPackageStrategiesTest {

    private static final String CLIENT = CampaignsLogins.CLIENT_NEW_FEATURES;
    private static final Integer DEFAULT_CONTEXT_LIMIT = 0;
    private static final Integer COUNTER_ID = 62927671;
    private static final Long GOAL_ID = 104567458L;

    private static Long clientId;
    private static DirectJooqDbSteps jooqDbSteps;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @BeforeClass
    @Step("Подготовим данные для теста")
    public static void beforeClass() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        int shard = api.userSteps.clientFakeSteps().getUserShard(CLIENT);
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShard(shard);
        clientId = jooqDbSteps.shardingSteps().getClientIdByLogin(CLIENT);
    }

    @Test
    public void defaultStrategyTest() {
        Long cid = api.userSteps.campaignSteps().addCampaign(new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withTextCampaign(new TextCampaignAddItemMap()
                        .withBiddingStrategy(new TextCampaignStrategyAddMap()
                                .withSearch(new TextCampaignSearchStrategyAddMap().defaultHighestPosition())
                                .withNetwork(new TextCampaignNetworkStrategyAddMap().defaultServingOff()))));
        CampaignsRecord campaignsRecord = jooqDbSteps.campaignsSteps().getCampaignById(cid);
        CampOptionsRecord campOptionsRecord = jooqDbSteps.campaignsSteps().getCampOptionsById(cid);
        Long strategyId = campaignsRecord.getStrategyId();
        StrategiesRecord strategiesRecord = jooqDbSteps.strategiesSteps().getStrategiesRecord(strategyId);
        StrategiesRecord expectedStrategiesRecord = new StrategiesRecord()
                .setStrategyId(strategyId)
                .setClientid(clientId)
                .setType(StrategiesType.default_)
                .setContextlimit(DEFAULT_CONTEXT_LIMIT)
                .setAttributionModel(StrategiesAttributionModel.last_yandex_direct_click_cross_device)
                .setDayBudget(BigDecimal.ZERO.setScale(2, RoundingMode.CEILING))
                .setDayBudgetShowMode(StrategiesDayBudgetShowMode.default_)
                .setWalletCid(campaignsRecord.getWalletCid())
                .setArchived(StrategiesArchived.No)
                .setDayBudgetDailyChangeCount(0)
                .setDayBudgetLastChange(campOptionsRecord.getDayBudgetLastChange())
                .setEnableCpcHold(StrategiesEnableCpcHold.Yes)
                .setName(null)
                .setMeaningfulGoals(null)
                .setIsPublic(StrategiesIsPublic.No);
        assertThat("стратегия правильно сохранилась в базе",
                strategiesRecord.intoMap(), beanDifferV5(expectedStrategiesRecord.intoMap())
                        .useCompareStrategy(DefaultCompareStrategies.allFieldsExcept(
                                BeanFieldPath.newPath(Strategies.STRATEGIES.STRATEGY_DATA.getName())
                        ).forFields(BeanFieldPath.newPath(Strategies.STRATEGIES.LASTCHANGE.getName()))
                                .useMatcher(not(before(campaignsRecord.getLastchange())))));
    }

    @Test
    public void strategyDataTest() {
        Long cid = api.userSteps.campaignSteps().addCampaign(new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withTextCampaign(new TextCampaignAddItemMap()
                        .withBiddingStrategy(new TextCampaignStrategyAddMap()
                                .withSearch(new TextCampaignSearchStrategyAddMap().defaultHighestPosition())
                                .withNetwork(new TextCampaignNetworkStrategyAddMap().defaultServingOff()))));
        CampaignsRecord campaignsRecord = jooqDbSteps.campaignsSteps().getCampaignById(cid);
        Long strategyId = campaignsRecord.getStrategyId();
        StrategiesRecord strategiesRecord = jooqDbSteps.strategiesSteps().getStrategiesRecord(strategyId);
        StrategyData actualStrategyData = StrategyData.fromString(strategiesRecord.getStrategyData());
        StrategyData expectedStrategyData = new StrategyData()
                .withName(CampaignsStrategyName.default_);
        assertThat("стратегия правильно сохранилась в базе",
                actualStrategyData, beanDifferV5(expectedStrategyData));
    }

    @Test
    public void notDefaultStrategyTest() {
        Long cid = api.userSteps.campaignSteps().addCampaign(new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withTextCampaign(new TextCampaignAddItemMap()
                        .withBiddingStrategy(new TextCampaignStrategyAddMap()
                                .withSearch(new TextCampaignSearchStrategyAddMap().defaultServingOff())
                                .withNetwork(new TextCampaignNetworkStrategyAddMap().defaultAverageCpc(Currency.RUB)))
                        .withSettings(new TextCampaignSettingMap()
                                .withOption(TextCampaignSettingsEnum.MAINTAIN_NETWORK_CPC)
                                .withValue(YesNoEnum.NO))));
        CampaignsRecord campaignsRecord = jooqDbSteps.campaignsSteps().getCampaignById(cid);
        CampOptionsRecord campOptionsRecord = jooqDbSteps.campaignsSteps().getCampOptionsById(cid);
        Long strategyId = campaignsRecord.getStrategyId();
        StrategiesRecord strategiesRecord = jooqDbSteps.strategiesSteps().getStrategiesRecord(strategyId);
        StrategiesRecord expectedStrategiesRecord = new StrategiesRecord()
                .setStrategyId(strategyId)
                .setClientid(clientId)
                .setType(StrategiesType.autobudget_avg_click)
                .setContextlimit(DEFAULT_CONTEXT_LIMIT)
                .setAttributionModel(StrategiesAttributionModel.last_yandex_direct_click_cross_device)
                .setDayBudget(BigDecimal.ZERO.setScale(2, RoundingMode.CEILING))
                .setDayBudgetShowMode(StrategiesDayBudgetShowMode.default_)
                .setWalletCid(campaignsRecord.getWalletCid())
                .setArchived(StrategiesArchived.No)
                .setDayBudgetDailyChangeCount(0)
                .setDayBudgetLastChange(campOptionsRecord.getDayBudgetLastChange())
                .setEnableCpcHold(StrategiesEnableCpcHold.No)
                .setName(null)
                .setMeaningfulGoals(null)
                .setIsPublic(StrategiesIsPublic.No);
        assertThat("стратегия правильно сохранилась в базе",
                strategiesRecord.intoMap(), beanDifferV5(expectedStrategiesRecord.intoMap())
                        .useCompareStrategy(DefaultCompareStrategies.allFieldsExcept(
                                BeanFieldPath.newPath(Strategies.STRATEGIES.STRATEGY_DATA.getName())
                        ).forFields(BeanFieldPath.newPath(Strategies.STRATEGIES.LASTCHANGE.getName()))
                                .useMatcher(not(before(campaignsRecord.getLastchange())))));
    }

    @Test
    public void meaningfulGoalsTest() {
        Long cid = api.userSteps.campaignSteps().addCampaign(new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withTextCampaign(new TextCampaignAddItemMap()
                        .withBiddingStrategy(new TextCampaignStrategyAddMap()
                                .withSearch(new TextCampaignSearchStrategyAddMap().defaultHighestPosition())
                                .withNetwork(new TextCampaignNetworkStrategyAddMap().defaultServingOff()))
                        .withCounterIds(COUNTER_ID)
                        .withPriorityGoals(new PriorityGoalsArrayMap()
                                .withItems(new PriorityGoalsItemMap()
                                        .withGoalId(GOAL_ID)
                                        .withValue(1_000_000L)))));
        CampaignsRecord campaignsRecord = jooqDbSteps.campaignsSteps().getCampaignById(cid);
        Long strategyId = campaignsRecord.getStrategyId();
        StrategiesRecord strategiesRecord = jooqDbSteps.strategiesSteps().getStrategiesRecord(strategyId);
        assertThat("meaningful_goals содержит правильный json",
                strategiesRecord.getMeaningfulGoals(),
                getMeaningfulGoalsJsonMatcher(GOAL_ID, 1L));
    }

    @Test
    public void strategyMetrikaCountersTest() {
        Long cid = api.userSteps.campaignSteps().addCampaign(new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withTextCampaign(new TextCampaignAddItemMap()
                        .withBiddingStrategy(new TextCampaignStrategyAddMap()
                                .withSearch(new TextCampaignSearchStrategyAddMap().defaultHighestPosition())
                                .withNetwork(new TextCampaignNetworkStrategyAddMap().defaultServingOff()))
                        .withCounterIds(COUNTER_ID)));
        CampaignsRecord campaignsRecord = jooqDbSteps.campaignsSteps().getCampaignById(cid);
        Long strategyId = campaignsRecord.getStrategyId();
        StrategyMetrikaCountersRecord strategyMetrikaCountersRecord =
                jooqDbSteps.strategyMetrikaCountersSteps().getStrategyMetrikaCountersRecord(strategyId);
        StrategyMetrikaCountersRecord expectedStrategyMetrikaCountersRecord = new StrategyMetrikaCountersRecord()
                .setStrategyId(strategyId)
                .setMetrikaCounter(COUNTER_ID.longValue())
                .setSource(StrategyMetrikaCountersSource.unknown)
                .setIsDeleted(0)
                .setHasEcommerce(0);
        assertThat("счетчики метрики стратегии правильно сохранилась в базе",
                strategyMetrikaCountersRecord.intoMap(), beanDifferV5(expectedStrategyMetrikaCountersRecord.intoMap()));
    }

    @Test
    public void shardIncStrategyIdTest() {
        Long cid = api.userSteps.campaignSteps().addCampaign(new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withTextCampaign(new TextCampaignAddItemMap()
                        .withBiddingStrategy(new TextCampaignStrategyAddMap()
                                .withSearch(new TextCampaignSearchStrategyAddMap().defaultHighestPosition())
                                .withNetwork(new TextCampaignNetworkStrategyAddMap().defaultServingOff()))));
        CampaignsRecord campaignsRecord = jooqDbSteps.campaignsSteps().getCampaignById(cid);
        Long strategyId = campaignsRecord.getStrategyId();
        ShardIncStrategyIdRecord actualRecord = jooqDbSteps.shardingSteps().getShardIncStrategyIdRecord(strategyId,
                clientId);
        ShardIncStrategyIdRecord expectedRecord = new ShardIncStrategyIdRecord()
                .setStrategyId(strategyId)
                .setClientid(clientId);
        assertThat("стратегия правильно сохранилась в базе",
                actualRecord.intoMap(), beanDifferV5(expectedRecord.intoMap()));
    }
}
