package ru.yandex.autotests.direct.api.campaigns.add.dailybudget;

import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.campaigns.AddResponse;
import com.yandex.direct.api.v5.campaigns.CampaignFieldEnum;
import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import com.yandex.direct.api.v5.campaigns.DailyBudgetModeEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.api5.campaigns.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignGetItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.DailyBudgetMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by pavryabov on 25.08.15.
 * https://st.yandex-team.ru/TESTIRT-6546
 */
@Aqua.Test
@Features(CampaignsFeatures.ADD)
@Description("Проверка валидации поля DailyBudget.amount")
@Issue("https://st.yandex-team.ru/DIRECT-34567")
@RunWith(Parameterized.class)
@Tag(TagDictionary.TRUNK)
public class DailyBudgetAmountTest {

    private static final Long MAX_DAILY_BUDGET = 1_000_000_000L;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(5);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter()
    public String login;

    @Parameterized.Parameter(1)
    public Currency currency;

    @Parameterized.Parameter(2)
    public String currencyInError;

    @Parameterized.Parameters(name = "Валюта проверки = {1}")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {Logins.LOGIN_FOR_RUB, Currency.RUB, MoneyCurrency.get(Currency.RUB).getAPIAbbreviation("en")},
                {Logins.LOGIN_FOR_USD, Currency.USD, MoneyCurrency.get(Currency.USD).getAPIAbbreviation("en")},
                {Logins.LOGIN_FOR_UAH, Currency.UAH, MoneyCurrency.get(Currency.UAH).getAbbreviation("en")},
                {Logins.LOGIN_FOR_EUR, Currency.EUR, MoneyCurrency.get(Currency.EUR).getAbbreviation("en")},
                {Logins.LOGIN_FOR_KZT, Currency.KZT, MoneyCurrency.get(Currency.KZT).getAPIAbbreviation("en")},
                {Logins.LOGIN_FOR_CHF, Currency.CHF, MoneyCurrency.get(Currency.CHF).getAbbreviation("en")},
                {Logins.LOGIN_FOR_TRY, Currency.TRY, MoneyCurrency.get(Currency.TRY).getAbbreviation("en")},
                {Logins.CLIENT_BYN, Currency.BYN, MoneyCurrency.get(Currency.BYN).getApiDescription()}
        });
    }

    @Before
    public void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(login);
        api.as(login);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("121")
    public void checkZeroAmount() {
        DailyBudgetMap dailyBudgetMap = new DailyBudgetMap()
                .withAmount(0L)
                .withMode(DailyBudgetModeEnum.STANDARD);
        api.userSteps.campaignSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withCampaigns(new CampaignAddItemMap()
                                .defaultCampaignAddItem().withDailyBudget(dailyBudgetMap)
                                .withDefaultTextCampaign()
                        ),
                ExpectedResult.errors(new Notification(5005,
                        Api5ErrorDetails.FIELD_MUST_BE_POSITIVE_INTEGER, DailyBudgetMap.AMOUNT_IN_ERROR)

                ));
    }

    @Test
    @Issue("https://st.yandex-team.ru/DIRECT-45503")
    @ru.yandex.qatools.allure.annotations.TestCaseId("122")
    public void checkMinimumAmount() {
        long amount = MoneyCurrency.get(currency).getMinDayBudgetAmount().bidLong().longValue();
        DailyBudgetMap dailyBudgetMap = new DailyBudgetMap()
                .withAmount(amount)
                .withMode(DailyBudgetModeEnum.STANDARD);
        AddResponse addResponse = api.userSteps.campaignSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withCampaigns(new CampaignAddItemMap()
                                .defaultCampaignAddItem().withDailyBudget(dailyBudgetMap)
                                .withDefaultTextCampaign()
                        ),
                ExpectedResult.success());
        Long id = addResponse.getAddResults().get(0).getId();
        CampaignGetItem campaignGetItem =
                api.userSteps.campaignSteps().campaignsGet(id, CampaignFieldEnum.DAILY_BUDGET);
        assertThat("поле DailyBudget сохранилось верно",
                campaignGetItem, beanDifferV5((CampaignGetItem)
                        new CampaignGetItemMap()
                                .withDailyBudget(dailyBudgetMap).getBean()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("123")
    public void lessThanMinimumAmount() {
        long amount = MoneyCurrency.get(currency).getMinDayBudgetAmount().getPrevious().bidLong().longValue();
        DailyBudgetMap dailyBudgetMap = new DailyBudgetMap()
                .withAmount(amount)
                .withMode(DailyBudgetModeEnum.STANDARD);
        api.userSteps.campaignSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withCampaigns(new CampaignAddItemMap()
                                .defaultCampaignAddItem().withDailyBudget(dailyBudgetMap)
                                .withDefaultTextCampaign()
                        ),
                ExpectedResult.errorsIgnoringDetails(new Notification(5005,
                        Api5ErrorDetails.DAILY_BUDGET_MINIMUM_WITH_CURRENCY,
                        MoneyCurrency.get(currency).getMinDayBudgetAmount()
                                .stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        currencyInError)));
    }

    @Test
    @Issue("https://st.yandex-team.ru/DIRECT-63508")
    public void lessThanMaximumAmount() {
        long amount = Money.valueOf(MAX_DAILY_BUDGET).getPrevious().bidLong().longValue();

        DailyBudgetMap dailyBudgetMap = new DailyBudgetMap()
                .withAmount(amount)
                .withMode(DailyBudgetModeEnum.STANDARD);
        AddResponse addResponse = api.userSteps.campaignSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withCampaigns(new CampaignAddItemMap()
                                .defaultCampaignAddItem().withDailyBudget(dailyBudgetMap)
                                .withDefaultTextCampaign()
                        ),
                ExpectedResult.success());

        Long id = addResponse.getAddResults().get(0).getId();

        CampaignGetItem campaignGetItem =
                api.userSteps.campaignSteps().campaignsGet(id, CampaignFieldEnum.DAILY_BUDGET);
        assertThat("поле DailyBudget сохранилось верно",
                campaignGetItem, beanDifferV5((CampaignGetItem)
                        new CampaignGetItemMap()
                                .withDailyBudget(dailyBudgetMap).getBean()));
    }

    @Test
    @Issue("https://st.yandex-team.ru/DIRECT-63508")
    public void checkMaximumAmount() {
        long amount = Money.valueOf(MAX_DAILY_BUDGET).bidLong().longValue();

        DailyBudgetMap dailyBudgetMap = new DailyBudgetMap()
                .withAmount(amount)
                .withMode(DailyBudgetModeEnum.STANDARD);
        AddResponse addResponse = api.userSteps.campaignSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withCampaigns(new CampaignAddItemMap()
                                .defaultCampaignAddItem().withDailyBudget(dailyBudgetMap)
                                .withDefaultTextCampaign()
                        ),
                ExpectedResult.success());

        Long id = addResponse.getAddResults().get(0).getId();

        CampaignGetItem campaignGetItem =
                api.userSteps.campaignSteps().campaignsGet(id, CampaignFieldEnum.DAILY_BUDGET);
        assertThat("поле DailyBudget сохранилось верно",
                campaignGetItem, beanDifferV5((CampaignGetItem)
                        new CampaignGetItemMap()
                                .withDailyBudget(dailyBudgetMap).getBean()));
    }

    @Test
    @Issue("https://st.yandex-team.ru/DIRECT-63508")
    public void moreThanMaximumAmount() {
        Money dailyBudget = Money.valueOf(MAX_DAILY_BUDGET);

        long amount = dailyBudget.getNext().bidLong().longValue();

        DailyBudgetMap dailyBudgetMap = new DailyBudgetMap()
                .withAmount(amount)
                .withMode(DailyBudgetModeEnum.STANDARD);
        api.userSteps.campaignSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withCampaigns(new CampaignAddItemMap()
                                .defaultCampaignAddItem().withDailyBudget(dailyBudgetMap)
                                .withDefaultTextCampaign()
                        ),
                ExpectedResult.errorsIgnoringDetails(new Notification(5005,
                        Api5ErrorDetails.DAILY_BUDGET_MAXIMUM_WITH_CURRENCY,
                        dailyBudget.stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        currencyInError)));
    }
}
