package ru.yandex.autotests.direct.api.campaigns.add.negativekeywords;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;

import com.yandex.direct.api.v5.campaigns.AddResponse;
import com.yandex.direct.api.v5.campaigns.CampaignFieldEnum;
import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import org.apache.commons.lang3.RandomStringUtils;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.campaigns.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignGetItemMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.banners.KeywordsSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * При переключении на Java нужно исправить тест
 *
 * Падение в Java ожидаемо, чиним в https://st.yandex-team.ru/DIRECT-156022
 */
@Aqua.Test
@Features(CampaignsFeatures.ADD)
@Description("Проверка валидации поля NegativeKeywords для минус-фраз. Позитивные сценарии")
@Issue("https://st.yandex-team.ru/TESTIRT-10419")
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class NegativeKeyPhrasesValidationPositiveTest {
    private static final String CLIENT = CampaignsLogins.CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public String[] negativePhrases;

    @Parameterized.Parameter(2)
    public String[] negativePhrasesAfterSave;

    @Parameterized.Parameters(name = "{0}")
    public static Collection testData() {
        String maxLengthNegativeWord = RandomStringUtils.randomAlphabetic(KeywordsSteps.MINUS_WORD_MAX_LENGTH);
        String maxWordInNegativePhrase = "polar bear sew fur coat !for winter";

        ArrayList<String> maxLengthNegativePhrasesList = new ArrayList<>();
        maxLengthNegativePhrasesList.add(maxWordInNegativePhrase);
        int allPhrasesLength = maxWordInNegativePhrase.length();
        while (allPhrasesLength <= (KeywordsSteps.PHRASE_MAX_LENGTH - KeywordsSteps.MINUS_WORD_MAX_LENGTH)) {
            allPhrasesLength = allPhrasesLength + KeywordsSteps.MINUS_WORD_MAX_LENGTH;
            maxLengthNegativePhrasesList.add(RandomStringUtils.randomAlphabetic(KeywordsSteps.MINUS_WORD_MAX_LENGTH));
        }
        if (KeywordsSteps.PHRASE_MAX_LENGTH - allPhrasesLength > 0) {
            maxLengthNegativePhrasesList.add(
                    RandomStringUtils.randomAlphabetic(KeywordsSteps.MINUS_WORD_MAX_LENGTH)
                            .substring(0, KeywordsSteps.PHRASE_MAX_LENGTH - allPhrasesLength));
        }
        Collections.sort(maxLengthNegativePhrasesList);
        Object[][] data = new Object[][]{
                {"Min 2 words for key phrase", new String[]{"белый медведь"}, new String[]{"белый медведь"}},
                {"Max 7 words for key phrase", new String[]{maxWordInNegativePhrase},
                        new String[]{"polar bear sew fur coat !for winter"}},
                {"Max length for word in negative key phrase",
                        new String[]{maxLengthNegativeWord + " белый медведь"},
                        new String[]{maxLengthNegativeWord + " белый медведь"}},
                {"Max length 4096 symbol for negative key phrase", maxLengthNegativePhrasesList.toArray(
                        new String[maxLengthNegativePhrasesList.size()]),
                        maxLengthNegativePhrasesList.toArray(new String[maxLengthNegativePhrasesList.size()])},
                // Поведение перла выглядит более правильно, разбираемся в https://st.yandex-team.ru/DIRECT-156022
//                {"Phrase with a dash", new String[]{"онлайн-переводчик"}, new String[]{"онлайн-переводчик"}},    // perl
                 {"Phrase with a dash", new String[]{"онлайн-переводчик"}, new String[]{"онлайн переводчик"}}, // java
                {"+в горшках цветы", new String[]{"+в горшках цветы"}, new String[]{"+в горшках цветы"}},
                {"цветы +в горшках", new String[]{"цветы +в горшках"}, new String[]{"цветы +в горшках"}},
                {"цветы +в горшках, кот в мешке", new String[]{"цветы +в горшках", "кот в мешке"},
                        new String[]{"кот !в мешке", "цветы +в горшках"}},
                {"купить !цветы", new String[]{"купить !цветы"},
                        new String[]{"купить !цветы"}},
                {"!купить !цветы", new String[]{"!купить !цветы"},
                        new String[]{"!купить !цветы"}},
                {"\"цветы в горшках\"", new String[]{"\"цветы в горшках\""},
                        new String[]{"\"цветы в горшках\""}},
                {"\"купить цветы +в горшках недорого\"", new String[]{"\"купить цветы +в горшках недорого\""},
                        new String[]{"\"купить цветы +в горшках недорого\""}},
                {"[из Москвы в Париж]", new String[]{"[из Москвы в Париж]"},
                        new String[]{"[из Москвы в Париж]"}},
                {"2 билета [из Москвы в Париж]", new String[]{"2 билета [из Москвы в Париж]"},
                        new String[]{"2 билета [из Москвы в Париж]"}},
                {"билеты [из Москвы в Париж]", new String[]{"билеты [из Москвы в Париж]"},
                        new String[]{"билеты [из Москвы в Париж]"}},
                {"[из !Москвы в Париж]", new String[]{"[из !Москвы в Париж]"},
                        new String[]{"[из !Москвы в Париж]"}},
                {" из !Москвы в Париж", new String[]{" из !Москвы в Париж"},
                        new String[]{"!из !Москвы !в Париж"}},
                {"\"[из Москвы в Париж]\"", new String[]{"\"[из Москвы в Париж]\""},
                        new String[]{"\"[из Москвы в Париж]\""}},
                // Поведение перла выглядит более правильно, разбираемся в https://st.yandex-team.ru/DIRECT-156022
//                {"говорить +по-английски", new String[]{"говорить +по-английски"},      // perl
//                        new String[]{"говорить +по-английски"}},
                {"говорить +по-английски", new String[]{"говорить +по-английски"},    // java
                        new String[]{"говорить +по +английски"}},
                // Поведение перла выглядит более правильно, разбираемся в https://st.yandex-team.ru/DIRECT-156022
//                {"[англо-русский переводчик]", new String[]{"[англо-русский переводчик]"}, // perl
//                        new String[]{"[англо-русский переводчик]"}},
                {"[англо-русский переводчик]", new String[]{"[англо-русский переводчик]"}, // java
                        new String[]{"[англо русский переводчик]"}}
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
    }

    @Test
    public void addWithNegativeKeyPhrasesAndCheckSave() {
        AddResponse addResponse = api.userSteps.campaignSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withCampaigns(new CampaignAddItemMap()
                                .defaultCampaignAddItem().withNegativeKeywords(negativePhrases)
                                .withDefaultTextCampaign()
                        ),
                ExpectedResult.success());
        Long id = addResponse.getAddResults().get(0).getId();
        CampaignGetItem campaignGetItem =
                api.userSteps.campaignSteps().campaignsGet(id, CampaignFieldEnum.NEGATIVE_KEYWORDS);
        assertThat("поле NegativeKeywords сохранилось верно", campaignGetItem,
                beanDifferV5((CampaignGetItem)
                        new CampaignGetItemMap()
                                .withNegativeKeywords(negativePhrasesAfterSave).getBean()));
    }
}
