package ru.yandex.autotests.direct.api.campaigns.archive;

import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsArchived;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusbssynced;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusshow;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampaignsRecord;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.BannerFakeInfo;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.GroupFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.common.Value;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by chicos on 25.08.15.
 */
@Aqua.Test
@Features(CampaignsFeatures.ARCHIVE)
@Description("Проверка статусов в БД при архивации кампании")
@Tag(TagDictionary.TRUNK)
public class ArchiveCampaignsDBStatusTest {
    private static LogSteps log = LogSteps.getLogger(ArchiveCampaignsDBStatusTest.class);
    private static final String client = CampaignsLogins.SINGLE_CLIENT_ARCHIVE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private static Long campaignID;
    private static Long groupID;
    private static Long adID;
    private static CampaignsRecord campaignDB;
    private static BannerFakeInfo adInfo;
    private static final DateTime lastChange = DateTime.now().minusDays(5);
    private static int shard;

    @Rule
    public Trashman trasher = new Trashman(api);

    @BeforeClass
    @Description("Подготовим данные для теста")
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        shard = api.userSteps.clientFakeSteps().getUserShard(client);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();
        groupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        adID = api.userSteps.adsSteps().addDefaultTextAd(groupID);
        api.userSteps.campaignSteps().campaignsSuspend(campaignID);

        log.info("Выставим статусы синхронизации с БК фейково");
        api.userSteps.campaignFakeSteps().setBSSynced(campaignID, true);
        api.userSteps.groupFakeSteps().setGroupFakeStatusBsSynced(groupID, Status.YES);
        api.userSteps.bannersFakeSteps().setStatusBsSynced(adID, Status.YES);

        log.info("Сдвинем время последнего изменения на несколько дней");
        api.userSteps.campaignFakeSteps().setLastChange(campaignID, lastChange.toString());
        api.userSteps.groupFakeSteps().setLastChange(groupID, lastChange.toString());
        api.userSteps.bannersFakeSteps().setLastChange(adID, lastChange.toString());

        log.info("Заархивируем кампанию");
        api.userSteps.campaignSteps().campaignsArchive(campaignID);
        campaignDB = api.userSteps.getDirectJooqDbSteps().useShard(shard).campaignsSteps().getCampaignById(campaignID);
        adInfo = api.userSteps.bannersFakeSteps().getBannerParams(adID);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("266")
    public void campaignStatusArchivedTest() {
        assertThat("поле Archived после архивации кампании", campaignDB.getArchived(), equalTo(CampaignsArchived.Yes));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("258")
    public void campaignStatusShowTest() {
        assertThat("поле StatusShow после архивации кампании", campaignDB.getStatusshow(),
                equalTo(CampaignsStatusshow.No));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("259")
    public void campaignBsSyncedTest() {
        assertThat("поле StatusBsSynced после архивации кампании", campaignDB.getStatusbssynced(),
                equalTo(CampaignsStatusbssynced.No));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("260")
    public void campaignLastChangeTest() {
        LocalDate date = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss.0")
                .parseLocalDate(campaignDB.getLastchange().toString());

        assertThat("время последнего изменения кампании после архивации кампании", date, equalTo(LocalDate.now()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("261")
    public void groupStatusShowsForecastTest() {
        GroupFakeInfo group = api.userSteps.groupFakeSteps().getGroupParams(groupID);

        assertThat("поле StatusShowsForecast после архивации кампании", group.getStatusShowsForecast(),
                equalTo("Archived"));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("262")
    public void groupStatusBsSyncedTest() {
        GroupFakeInfo group = api.userSteps.groupFakeSteps().getGroupParams(groupID);

        assertThat("поле StatusBsSynced после архивации кампании", group.getStatusBsSynced(), equalTo(Value.YES));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("263")
    public void adStatusBsSyncedTest() {
        assertThat("поле StatusBsSynced после архивации кампании", adInfo.getStatusBsSynced(), equalTo(Value.YES));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("264")
    public void adStatusActiveTest() {
        assertThat("поле StatusActive после архивации кампании", adInfo.getStatusActive(), equalTo(Value.NO));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("265")
    public void adLastChangeTest() {
        LocalDate date = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss").parseLocalDate(adInfo.getLastChange());

        assertThat("время последнего изменения объявления после архивации кампании", date, equalTo(LocalDate.now()));
    }
}
