package ru.yandex.autotests.direct.api.campaigns.archive;

import com.yandex.direct.api.v5.campaigns.CampaignFieldEnum;
import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import com.yandex.direct.api.v5.campaigns.CampaignStateGetEnum;
import com.yandex.direct.api.v5.general.StatusEnum;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.campaigns.ArchiveRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 20.08.15.
 */
@Aqua.Test
@Features(CampaignsFeatures.ARCHIVE)
@Description("Проверка негативных сценариев архивирования кампаний")
@Tag(TagDictionary.TRUNK)
public class ArchiveCampaignsNegativeTest {
    private final LogSteps log = LogSteps.getLogger(this.getClass());
    private static final String CLIENT = CampaignsLogins.SINGLE_CLIENT_ARCHIVE;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    private static Long unsuspendedDraftCampaign;
    private static Long unsuspendedModeratedCampaign;
    private static Long campaignArchived;
    private static Long campaignWithMoney;
    private static Long campaignSuspended;
    private static Long campaignID1;
    private static Long campaignID2;
    private static final Currency currency = Currency.RUB;
    private static final Money sumToAdd = MoneyCurrency.get(currency).getMinInvoiceAmount();

    @Rule
    public Trashman trasher = new Trashman(api);

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);

        unsuspendedDraftCampaign = api.userSteps.campaignSteps().addDefaultTextCampaign();
        unsuspendedModeratedCampaign = api.userSteps.campaignSteps().addDefaultTextCampaign();
        campaignSuspended = api.userSteps.campaignSteps().addDefaultTextCampaign();
        campaignID1 = api.userSteps.campaignSteps().addDefaultTextCampaign();
        campaignID2 = api.userSteps.campaignSteps().addDefaultTextCampaign();
        campaignArchived = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignSteps().campaignsSuspend(campaignArchived, campaignSuspended, campaignID1, campaignID2);
        api.userSteps.campaignSteps().campaignsArchive(campaignArchived);

        campaignWithMoney = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignFakeSteps().makeCampaignModerated(campaignWithMoney);
        api.userSteps.campaignFakeSteps().makeCampaignModerated(unsuspendedModeratedCampaign);
        CampaignGetItem campaign =
                api.userSteps.campaignSteps().campaignsGet(campaignWithMoney, CampaignFieldEnum.STATUS);
        assumeThat("кампания фейково промодерирована", campaign.getStatus(), equalTo(StatusEnum.ACCEPTED));
        api.userSteps.campaignFakeSteps().sendFakeNotificationFromBalance(
                campaignWithMoney,
                sumToAdd.addVAT().floatValue(),
                currency);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("269")
    public void archiveArchivedCampaignTest() {
        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.ARCHIVE,
                new ArchiveRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignArchived)),
                ExpectedResult.warnings(campaignArchived,
                        new Notification(10022, Api5ErrorDetails.CAMPAIGN_ALREADY_ARCHIVED)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("270")
    public void archiveUnsuspendedCampaignsTest() {
        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.ARCHIVE,
                new ArchiveRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(unsuspendedDraftCampaign, unsuspendedModeratedCampaign)),
                ExpectedResult.success(unsuspendedDraftCampaign),
                ExpectedResult
                        .errors(new Notification(8303, Api5ErrorDetails.CAMPAIGN_MUST_BE_STOPPED_BEFORE_ARCHIVATION)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("271")
    public void archiveCampaignWithMoneyTest() {
        log.info("Попробуем заархивировать кампанию с деньгами");
        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.ARCHIVE,
                new ArchiveRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignWithMoney)),
                ExpectedResult.errors(new Notification(8303, Api5ErrorDetails.COULD_NOT_ARCHIVE_CAMPAIGN_WITH_MONEY)));
    }

    @Test
    @Description("Архивация двух кампаний, идентификатор одной из которых - некорректен")
    @ru.yandex.qatools.allure.annotations.TestCaseId("272")
    public void suspendMultipleCampaignsOneInvalidTest() {
        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.ARCHIVE,
                new ArchiveRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignSuspended, 123456L)),
                ExpectedResult.success(campaignSuspended),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)));
    }

    @Test
    @Description("Два однинаковых идентификатора кампании в запросе")
    @ru.yandex.qatools.allure.annotations.TestCaseId("273")
    public void sameIdsInRequestTest() {
        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.ARCHIVE,
                new ArchiveRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignID1, campaignID1, campaignID2)),
                ExpectedResult.warnings(
                        new Notification(10000, Api5ErrorDetails.CAMPAIGN_ID_IS_PRESENTED_MORE_THAN_ONCE_IN_REQUEST)),
                ExpectedResult.warnings(
                        new Notification(10000, Api5ErrorDetails.CAMPAIGN_ID_IS_PRESENTED_MORE_THAN_ONCE_IN_REQUEST)),
                ExpectedResult.success(campaignID2));

        CampaignGetItem campaign = api.userSteps.campaignSteps().campaignsGet(campaignID1, CampaignFieldEnum.STATE);
        assertThat("кампания заархивирована несмотря на warning", campaign.getState(),
                equalTo(CampaignStateGetEnum.ARCHIVED));
    }

    @Test
    @Description("Попробуем заархивировать кампанию, у которой время последнего показа менее часа")
    @ru.yandex.qatools.allure.annotations.TestCaseId("274")
    public void archiveCampaignLastShowLessThanAnHourTest() {
        //DIRECT-45550
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignFakeSteps().makeCampaignStopped(campaignID);
        api.userSteps.campaignFakeSteps().setLastShowTimeNow(campaignID);

        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.ARCHIVE,
                new ArchiveRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignID)),
                ExpectedResult
                        .errors(new Notification(8303, Api5ErrorDetails.CAMPAIGN_MUST_BE_STOPPED_BEFORE_ARCHIVATION)));
    }
}
