package ru.yandex.autotests.direct.api.campaigns.archive;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.campaigns.ArchiveRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 20.08.15.
 */
@Aqua.Test
@Features(CampaignsFeatures.ARCHIVE)
@Stories(ApiStories.UNITS)
@Description("Списание баллов при архивировании кампании")
public class ArchiveCampaignsUnitsTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());
    private static final String client = CampaignsLogins.CLIENT_UNITS_ARCHIVE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final int ARCHIVE_COST = 10;
    private static final int ITEM_COST_ON_ARCHIVE = 5;
    private static final int ITEM_COST_ON_ARCHIVE_FAULT = 20;
    private static final int GENERAL_FAULT_COST = 50;

    private static final Long INVALID_ID = 123456L;
    private static final Long INVALID_ID_2 = 123457L;

    @Before
    public void clearUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
    }

    @Test
    @Description("Архивирование одиночной кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("293")
    public void archiveSingleCampaignTest() {
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignSteps().campaignsSuspend(campaignID);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().campaignsArchive(campaignID);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость архивирования одной кампании", unitsBefore - unitsAfter,
                equalTo(ARCHIVE_COST + ITEM_COST_ON_ARCHIVE));
    }

    @Test
    @Description("Архивирование нескольких кампаний")
    @ru.yandex.qatools.allure.annotations.TestCaseId("297")
    public void archiveSeveralCampaignsTest() {
        Long campaignID1 = api.userSteps.campaignSteps().addDefaultTextCampaign();
        Long campaignID2 = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignSteps().campaignsSuspend(campaignID1, campaignID2);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().campaignsArchive(campaignID1, campaignID2);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость архивирования двух кампаний", unitsBefore - unitsAfter,
                equalTo(ARCHIVE_COST + ITEM_COST_ON_ARCHIVE * 2));
    }

    @Test
    @Description("Архивирование одной невалидной кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("298")
    public void archiveInvalidCampaignTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.ARCHIVE,
                new ArchiveRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(INVALID_ID)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при архивировании кампании", unitsBefore - unitsAfter,
                equalTo(ARCHIVE_COST + ITEM_COST_ON_ARCHIVE_FAULT));
    }

    @Test
    @Description("Архивирование двух невалидных кампаний")
    @ru.yandex.qatools.allure.annotations.TestCaseId("299")
    public void archiveFewInvalidCampaignsTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.ARCHIVE,
                new ArchiveRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(INVALID_ID, INVALID_ID_2)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при архивировании двух кампаний", unitsBefore - unitsAfter,
                equalTo(ARCHIVE_COST + ITEM_COST_ON_ARCHIVE_FAULT * 2));
    }

    @Test
    @Description("Архивирование одной корректной и одной невалидной кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("294")
    public void archiveOneValidAndOneInvalidCampaignsTest() {
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignSteps().campaignsSuspend(campaignID);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.ARCHIVE,
                new ArchiveRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignID, INVALID_ID)),
                ExpectedResult.success(campaignID),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при архивировании нескольких кампаний",
                unitsBefore - unitsAfter, equalTo(ARCHIVE_COST + ITEM_COST_ON_ARCHIVE + ITEM_COST_ON_ARCHIVE_FAULT));
    }

    @Test
    @Description("Невалидный запрос на архивирование кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("295")
    public void generalErrorOnArchiveCampaignTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().shouldGetCampaignErrorOn(
                Action.ARCHIVE,
                new ArchiveRequestMap().withSelectionCriteria(null),
                new Api5Error(8000, Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                        "params",
                        capitalize(ArchiveRequestMap.SELECTION_CRITERIA)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость общей ошибки при архивировании кампании", unitsBefore - unitsAfter,
                equalTo(GENERAL_FAULT_COST));
    }

    @Test
    @Description("Недостаточное колличество баллов при архивировании кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("296")
    public void notEnoughUnitsOnArchiveCampaignTest() {
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignSteps().campaignsSuspend(campaignID);

        log.info("Обнулим колличество доступных пользователю баллов");
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(client, unitsBefore);
        unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assumeThat("доступные баллы пользователя", unitsBefore, equalTo(0));

        api.userSteps.campaignSteps().shouldGetCampaignErrorOn(
                Action.ARCHIVE,
                new ArchiveRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignID)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("доступные баллы пользователя после ошибки", unitsAfter, equalTo(0));
    }
}
