package ru.yandex.autotests.direct.api.campaigns.delete;

import java.sql.Timestamp;
import java.util.List;

import com.yandex.direct.api.v5.campaigns.CampaignFieldEnum;
import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import com.yandex.direct.api.v5.campaigns.CampaignStateEnum;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BsExportQueueRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.common.api45mng.CreateNewSubclientResponse;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.DeleteRequestMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.clients.ConvertType;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 20.08.15.
 */
@Aqua.Test
@Features(CampaignsFeatures.DELETE)
@Description("Проверка негативных сценариев удаления кампаний")
@Tag(TagDictionary.TRUNK)
public class DeleteCampaignsNegativeTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    private static final String client = CampaignsLogins.SINGLE_CLIENT_DELETE;
    private static final String clientMoney = Logins.CLIENT_FOR_RUB;
    private static final String clientAccount = CampaignsLogins.SINGLE_CLIENT_DELETE_ACCOUNT;
    private static final Currency currency = Currency.RUB;
    private final DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss.0");

    private static Long campaignID;
    private static Long campaignID1;
    private static Long campaignID2;

    private static Long campaignExportQueue;
    private static String srvClient;

    private static DirectJooqDbSteps jooqDbSteps;

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void prepareCampaigns() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(clientMoney);

        //Создадим сервисируемого клиента YE
        CreateNewSubclientResponse createNewSubclientResponse =
                api.as(Logins.MANAGER_GEO).wsdl(ru.yandex.autotests.directapi.common.api45mng.APIPort_PortType.class)
                        .userSteps.clientSteps().createServicedClient("subClient", Logins.MANAGER_GEO);
        assumeThat("клиент создан успешно", createNewSubclientResponse.getLogin(), notNullValue());
        srvClient = createNewSubclientResponse.getLogin();
        api.userSteps.clientFakeSteps().enableAPI(srvClient);

        api.as(client).wsdl(ru.yandex.autotests.directapi.common.api45.APIPort_PortType.class);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();
        campaignID1 = api.userSteps.campaignSteps().addDefaultTextCampaign();
        campaignID2 = api.userSteps.campaignSteps().addDefaultTextCampaign();

        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(clientMoney);
    }

    @Test
    @Description("Попробуем удалить кампанию отправленную в очередь синхронизации с БК")
    @ru.yandex.qatools.allure.annotations.TestCaseId("341")
    public void campaignInExportMasterQueueTest() {
        log.info("Создадим кампанию");
        campaignExportQueue = api.as(clientMoney).userSteps.campaignSteps().addDefaultTextCampaign();

        log.info("Сбросим фейково статус синхронизации с БК");
        api.userSteps.campaignFakeSteps().setBSSynced(campaignExportQueue, false);

        log.info("Добавим кампанию в таблицу очереди на синхронизацию с БК");
        Timestamp now = new Timestamp(System.currentTimeMillis());
        jooqDbSteps.bsExportQueueSteps().createBsExportQueue(
                new BsExportQueueRecord()
                        .setCid(campaignExportQueue)
                        .setSeqTime(now)
                        .setQueueTime(now)
                        .setParId(2)
                        .setSyncVal(0L)
                        .setCampsNum(1L)
                        .setBannersNum(0L)
                        .setContextsNum(0L)
                        .setBidsNum(0L)
                        .setPricesNum(0L)
        );

        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.DELETE,
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignExportQueue)),
                ExpectedResult.errors(new Notification(8301, Api5ErrorDetails.UNABLE_TO_DELETE_CAMPAIGN)));
    }

    @Test
    @Description("Попробуем удалить кампанию, у которой есть ордер в БК")
    @ru.yandex.qatools.allure.annotations.TestCaseId("342")
    public void campaignWithBsOrderIdTest() {
        log.info("Создадим кампанию");
        Long campaignId = api.as(clientMoney).userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignFakeSteps().setRandomOrderID(campaignId);

        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.DELETE,
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignId)),
                ExpectedResult
                        .errors(new Notification(8301, Api5ErrorDetails.CAMPAIGN_WITH_MONEY_COULD_NOT_BE_DELETED)));
    }

    @Test
    @Description("Попробуем удалить кампанию с деньгами")
    @ru.yandex.qatools.allure.annotations.TestCaseId("343")
    public void campaignWithMoneyTest() {
        log.info("Создадим активную кампанию");
        Long campaignId = api.as(clientMoney).userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignFakeSteps().setCampaignSum(campaignId,
                MoneyCurrency.get(currency).getMinInvoiceAmount().addVAT().floatValue());

        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.DELETE,
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignId)),
                ExpectedResult
                        .errors(new Notification(8301, Api5ErrorDetails.CAMPAIGN_WITH_MONEY_COULD_NOT_BE_DELETED)));
    }

    @Test
    @Description("Попробуем удалить кампанию с выставленным на оплату счётом")
    @ru.yandex.qatools.allure.annotations.TestCaseId("344")
    public void campaignWithInvoiceTest() {
        Long campaignID = api.as(clientMoney).userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignFakeSteps().makeCampaignModerated(campaignID);
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID.intValue())
                        .withCurrency(currency)
                        .withSum(MoneyCurrency.get(currency).getMinInvoiceAmount().addVAT().floatValue()));
        assumeThat("выставлен счет на оплату", requestID, notNullValue());

        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.DELETE,
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignID)),
                ExpectedResult
                        .errors(new Notification(8301, Api5ErrorDetails.CAMPAIGN_WITH_MONEY_COULD_NOT_BE_DELETED)));
    }

    @Test
    @Description("Удаление двух кампаний, идентификатор одной из которых - некорректен")
    @ru.yandex.qatools.allure.annotations.TestCaseId("345")
    public void deleteMultipleCampaignsOneInvalidTest() {
        //DIRECT-45309
        api.as(client).userSteps.campaignSteps().shouldGetResultOn(
                Action.DELETE,
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignID, 123456L)),
                ExpectedResult.success(campaignID),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)));
    }

    @Test
    @Description("Два однинаковых идентификатора кампании в запросе")
    @ru.yandex.qatools.allure.annotations.TestCaseId("340")
    public void sameIdsInRequestTest() {
        api.as(client).userSteps.campaignSteps().shouldGetResultOn(
                Action.DELETE,
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignID1, campaignID1, campaignID2)),
                ExpectedResult.errors(new Notification(9800,
                        Api5ErrorDetails.CAMPAIGN_ID_IS_PRESENTED_MORE_THAN_ONCE_IN_REQUEST)),
                ExpectedResult.errors(new Notification(9800,
                        Api5ErrorDetails.CAMPAIGN_ID_IS_PRESENTED_MORE_THAN_ONCE_IN_REQUEST)),
                ExpectedResult.success(campaignID2));
    }

    @Test
    @Description("Попытка удаления общего счета вместо кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("346")
    public void accountInsteadOfCampaignTest() {
        Integer accountID = api.as(clientAccount).userSteps.financeSteps().getAccountID(clientAccount);
        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.DELETE,
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(accountID.longValue())),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)));
    }

    @Test
    @Description("Проверка возможности удаления кампании созданной до конвертации COPY")
    @ru.yandex.qatools.allure.annotations.TestCaseId("347")
    @Ignore
    public void deleteOldCampaignAfterCopyConvertTest() {
        Long oldCampaignID = api.as(Logins.MANAGER_GEO).userSteps.campaignSteps().addDefaultTextCampaign(srvClient);

        log.info("Сконвертируем клиента методом COPY");
        api.userSteps.clientFakeSteps().convertCurrency(
                srvClient,
                currency.toString(),
                ConvertType.COPY);

        List<CampaignGetItem> campaignInfos = api.userSteps.campaignSteps().getCampaigns(srvClient,
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap().withStates(CampaignStateEnum.SUSPENDED))
                        .withFieldNames(CampaignFieldEnum.ID));
        assumeThat("у клиента есть одна новая кампания", campaignInfos, hasSize(1));
        Long newCampaignID = campaignInfos.get(0).getId();

        log.info("Попробуем удалить старую фишковую кампанию");
        api.userSteps.campaignSteps().shouldGetResultOn(
                srvClient,
                Action.DELETE,
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(oldCampaignID)),
                ExpectedResult.errors(new Notification(8300, Api5ErrorDetails.UNABLE_UPDATE_ARCHIVED_CAMPAGN)));

        log.info("Удалим новую рублёвую кампанию");
        api.userSteps.campaignSteps().shouldGetResultOn(
                srvClient,
                Action.DELETE,
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(newCampaignID)),
                ExpectedResult.success(newCampaignID));
    }

    @AfterClass
    public static void prepareCampaignToRemove() {
        if (campaignExportQueue != null) {
            jooqDbSteps.bsExportQueueSteps().deleteBsExportQueue(campaignExportQueue);
        }
    }
}
