package ru.yandex.autotests.direct.api.campaigns.delete;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.Account;
import ru.yandex.autotests.directapi.common.api45.AccountManagementResponse;
import ru.yandex.autotests.directapi.common.api45.CreateNewSubclientResponse;
import ru.yandex.autotests.directapi.matchers.axiserror.ApiResponse;
import ru.yandex.autotests.directapi.model.finances.sharedaccount.AccountSelectionCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by chicos on 24.08.15.
 */
@Aqua.Test
@Features(CampaignsFeatures.DELETE)
@Description("Проверка позитивных сценариев удаления кампаний")
@Tag(TagDictionary.TRUNK)
public class DeleteCampaignsSharedAccountWithMoneyTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());
    private static final String agencyLogin = CampaignsLogins.AGENCY_RUB;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(agencyLogin);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long campaignID;

    private static String subClient;
    private static Integer accountID;
    private static final Currency currency = Currency.RUB;
    private static final Money sumToAdd = MoneyCurrency.get(currency).getMinInvoiceAmount();

    @BeforeClass
    public static void prepareCampaigns() {
        api.userSteps.clientFakeSteps().enableToCreateSubClients(agencyLogin);
        api.userSteps.clientFakeSteps().enableAgencyWallet(agencyLogin);
    }

    @Before
    @Step("Подготовим данные для теста")
    public void createSubclientWithAccount() {
        log.info("Создадим субклиента с ОС и одной кампанией");
        CreateNewSubclientResponse createNewSubclientResponse =
                api.userSteps.clientSteps().createNewAgencySubClient("subClient", agencyLogin, currency);
        subClient = createNewSubclientResponse.getLogin();
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(subClient);
        accountID = api.userSteps.financeSteps().enableAndGetSharedAccount(subClient);
        api.userSteps.campaignFakeSteps().resetWalletCreateDate(campaignID);

        log.info("Пополним ОС минимальной суммой");
        api.userSteps.payAccountWithBalance(subClient, sumToAdd.addVAT().floatValue(), currency);

        log.info("Дождемся появления средств на ОС");
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.financeSteps().accountSumChanged(accountID));

        log.info("Удалим единственную кампанию");
        api.userSteps.campaignSteps().campaignsDelete(subClient, campaignID);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("353")
    public void expectErrorGetAccountWithoutCampaignsUseAccountID() {
        api.userSteps.financeSteps().shouldGetErrorsOnAccountManagementGet(
                accountID, new AxisError(54, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("352")
    public void expectErrorGetAccountWithoutCampaignsLoginTest() {
        AccountManagementResponse response = api.userSteps.financeSteps().get(
                new AccountSelectionCriteriaMap(api.type()).withLogins(subClient));
        assertThat("получили ошибку при запросе ОС без кампаний", JsonUtils.toString(response), ApiResponse.hasError(
                new AxisError(515, AxisErrorDetails.EMPTY_STRING)
        ));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("355")
    public void checkAccountIdAfterReconnectTest() {
        api.userSteps.campaignSteps().addDefaultTextCampaign(subClient);
        int newAccountID = api.userSteps.financeSteps().getAccountID(subClient);
        assertThat("после повторного включения ОС AccountID не изменился", newAccountID, equalTo(accountID));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("354")
    public void checkAmountAfterAccountReconnectTest() {
        api.userSteps.campaignFakeSteps().setCampaignSum(accountID, 0f);
        api.userSteps.campaignSteps().addDefaultTextCampaign(subClient);
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.financeSteps().accountSumChanged(accountID));
        Account account = api.userSteps.financeSteps().getAccount(accountID);
        assertThat("после повторного включения ОС поле Amount у кошелька не изменилось",
                account.getAmount(), equalTo(sumToAdd.floatValue()));
    }
}
