package ru.yandex.autotests.direct.api.campaigns.delete;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.campaigns.DeleteRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 20.08.15.
 */
@Aqua.Test
@Features(CampaignsFeatures.DELETE)
@Stories(ApiStories.UNITS)
@Description("Списание баллов при удалении кампании")
public class DeleteCampaignsUnitsTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());
    private static final String client = CampaignsLogins.CLIENT_UNITS_DELETE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final int DELETE_COST = 10;
    private static final int ITEM_COST_ON_DELETE = 2;
    private static final int ITEM_COST_ON_DELETE_FAULT = 20;
    private static final int GENERAL_FAULT_COST = 50;

    private static final Long INVALID_ID = 123456L;
    private static final Long INVALID_ID_2 = 123457L;

    @Before
    public void clearUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
    }

    @Test
    @Description("Удаление одиночной кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("373")
    public void deleteSingleCampaignTest() {
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().campaignsDelete(campaignID);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость удаления одной кампании", unitsBefore - unitsAfter,
                equalTo(DELETE_COST + ITEM_COST_ON_DELETE));
    }

    @Test
    @Description("Удаление нескольких кампаний")
    @ru.yandex.qatools.allure.annotations.TestCaseId("368")
    public void deleteSeveralCampaignsTest() {
        Long campaignID1 = api.userSteps.campaignSteps().addDefaultTextCampaign();
        Long campaignID2 = api.userSteps.campaignSteps().addDefaultTextCampaign();

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().campaignsDelete(campaignID1, campaignID2);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость удаления двух кампаний", unitsBefore - unitsAfter,
                equalTo(DELETE_COST + ITEM_COST_ON_DELETE * 2));
    }

    @Test
    @Description("Удаление одной невалидной кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("369")
    public void deleteInvalidCampaignTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.DELETE,
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(INVALID_ID)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при удалении кампании", unitsBefore - unitsAfter,
                equalTo(DELETE_COST + ITEM_COST_ON_DELETE_FAULT));
    }

    @Test
    @Description("Удаление двух невалидных кампаний")
    @ru.yandex.qatools.allure.annotations.TestCaseId("370")
    public void deleteFewInvalidCampaignsTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.DELETE,
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(INVALID_ID, INVALID_ID_2)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при удалении двух кампаний", unitsBefore - unitsAfter,
                equalTo(DELETE_COST + ITEM_COST_ON_DELETE_FAULT * 2));
    }

    @Test
    @Description("Удаление одной корректной и одной невалидной кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("371")
    public void deleteOneValidAndOneInvalidCampaignsTest() {
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.DELETE,
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignID, INVALID_ID)),
                ExpectedResult.success(campaignID),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при удалении нескольких кампаний",
                unitsBefore - unitsAfter, equalTo(DELETE_COST + ITEM_COST_ON_DELETE + ITEM_COST_ON_DELETE_FAULT));
    }

    @Test
    @Description("Невалидный запрос на удаление кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("372")
    public void generalErrorOnDeleteCampaignTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().shouldGetCampaignErrorOn(
                Action.DELETE,
                new DeleteRequestMap().withSelectionCriteria(null),
                new Api5Error(8000, Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                        "params",
                        capitalize(DeleteRequestMap.SELECTION_CRITERIA)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость общей ошибки при удалении кампании", unitsBefore - unitsAfter,
                equalTo(GENERAL_FAULT_COST));
    }

    @Test
    @Description("Недостаточное колличество баллов при удалении кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("374")
    public void notEnoughUnitsOnDeleteCampaignTest() {
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();

        log.info("Обнулим колличество доступных пользователю баллов");
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(client, unitsBefore);
        unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assumeThat("доступные баллы пользователя", unitsBefore, equalTo(0));

        api.userSteps.campaignSteps().shouldGetCampaignErrorOn(
                Action.DELETE,
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignID)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("доступные баллы пользователя после ошибки", unitsAfter, equalTo(0));
    }
}
