package ru.yandex.autotests.direct.api.campaigns.get;

import java.util.Arrays;
import java.util.List;

import com.yandex.direct.api.v5.campaigns.CampaignFieldEnum;
import com.yandex.direct.api.v5.campaigns.GetResponse;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.LimitOffsetMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.extractProperty;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by ginger on 22.07.15.
 * https://st.yandex-team.ru/TESTIRT-6407
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-34568")
@Features(CampaignsFeatures.GET)
@Description("Проверка логики полей Limit и Offset в Page")
@Tag(TagDictionary.TRUNK)
public class GetPageTest {
    private static String client = CampaignsLogins.GET_CLIENT;
    private static String anotherClient = CampaignsLogins.CLIENT_SELF;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(5).as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long[] ids;
    private static Long anotherClientCampID;

    @BeforeClass
    public static void createCampaigns() {
        anotherClientCampID = api.as(anotherClient).userSteps.campaignSteps().addDefaultTextCampaign();
        ids = api.as(client).userSteps.campaignSteps().addDefaultTextCampaigns(4).toArray(new Long[]{});
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("440")
    public void getHead() {
        GetResponse getResponse = api.userSteps.campaignSteps().campaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap()
                                        .withIds(
                                                ids
                                        )
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(2L)
                                        .withOffset(0L)));
        List<Long> gotids = extractProperty(getResponse.getCampaigns(), CampaignFieldEnum.ID.value());
        assertThat("вернулся правильный набор кампаний", gotids, equalTo(Arrays.asList(ids[0], ids[1])));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("441")
    public void getBody() {
        GetResponse getResponse = api.userSteps.campaignSteps().campaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap()
                                        .withIds(
                                                ids
                                        )
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(2L)
                                        .withOffset(1L)));
        List<Long> gotids = extractProperty(getResponse.getCampaigns(), CampaignFieldEnum.ID.value());
        assertThat("вернулся правильный набор кампаний", gotids, equalTo(Arrays.asList(ids[1], ids[2])));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("439")
    public void getTail() {
        GetResponse getResponse = api.userSteps.campaignSteps().campaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap()
                                        .withIds(
                                                ids
                                        )
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(2L)
                                        .withOffset(2L)));
        List<Long> gotids = extractProperty(getResponse.getCampaigns(), CampaignFieldEnum.ID.value());
        assertThat("вернулся правильный набор кампаний", gotids, equalTo(Arrays.asList(ids[2], ids[3])));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("443")
    public void getPartOfTail() {
        GetResponse getResponse = api.userSteps.campaignSteps().campaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap()
                                        .withIds(
                                                ids
                                        )
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(2L)
                                        .withOffset(3L)));
        List<Long> gotids = extractProperty(getResponse.getCampaigns(), CampaignFieldEnum.ID.value());
        assertThat("вернулся правильный набор кампаний", gotids, equalTo(Arrays.asList(ids[3])));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("444")
    public void getOverTail() {
        GetResponse getResponse = api.userSteps.campaignSteps().campaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap()
                                        .withIds(
                                                ids
                                        )
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(2L)
                                        .withOffset(4L)));
        assertThat("вернулся правильный набор кампаний", getResponse.getCampaigns(), hasSize(0));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("445")
    public void onlyLimit() {
        GetResponse getResponse = api.userSteps.campaignSteps().campaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap()
                                        .withIds(
                                                ids
                                        )
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(2L)));
        List<Long> gotids = extractProperty(getResponse.getCampaigns(), CampaignFieldEnum.ID.value());
        assertThat("вернулся правильный набор кампаний", gotids, equalTo(Arrays.asList(ids[0], ids[1])));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("446")
    public void onlyOffset() {
        GetResponse getResponse = api.userSteps.campaignSteps().campaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap()
                                        .withIds(
                                                ids
                                        )
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withOffset(2L)));
        List<Long> gotids = extractProperty(getResponse.getCampaigns(), CampaignFieldEnum.ID.value());
        assertThat("вернулся правильный набор кампаний", gotids, equalTo(Arrays.asList(ids[2], ids[3])));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("447")
    public void zeroLimit() {
        api.userSteps.campaignSteps().expectErrorOnCampaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap()
                                        .withIds(
                                                ids
                                        )
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(0L)),
                new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("448")
    public void negativeLimit() {
        api.userSteps.campaignSteps().expectErrorOnCampaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap()
                                        .withIds(
                                                ids
                                        )
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(-1L)),
                new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("442")
    public void maxLimit() {
        GetResponse getResponse = api.userSteps.campaignSteps().campaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap()
                                        .withIds(
                                                ids
                                        )
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(LimitOffsetMap.LIMIT_MAX_LENGTH)));
        List<Long> gotids = extractProperty(getResponse.getCampaigns(), CampaignFieldEnum.ID.value());
        assertThat("вернулся правильный набор кампаний", gotids, equalTo(Arrays.asList(ids)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("449")
    public void overMaxLimit() {
        api.userSteps.campaignSteps().expectErrorOnCampaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap()
                                        .withIds(
                                                ids
                                        )
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(LimitOffsetMap.LIMIT_MAX_LENGTH + 1)),
                new Api5Error(4002, Api5ErrorDetails.LIMIT_CANNOT_BE_MORE, LimitOffsetMap.LIMIT_MAX_LENGTH));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("450")
    public void negativeOffset() {
        api.userSteps.campaignSteps().expectErrorOnCampaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap()
                                        .withIds(
                                                ids
                                        )
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withOffset(-1L)),
                new Api5Error(4002, Api5ErrorDetails.OFFSET_MUST_BE_MORE_OR_EQUAL_THAN_0));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("451")
    public void checkLimitedByIfLimitDidntWork() {
        GetResponse getResponse = api.userSteps.campaignSteps().campaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap()
                                        .withIds(
                                                ids
                                        )
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(4L)));
        assertThat("значение LimitedBy верно", getResponse.getLimitedBy(), nullValue());
    }

    @Issue("DIRECT-44597")
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("452")
    public void checkLimitedByIfLimitWorked() {
        GetResponse getResponse = api.userSteps.campaignSteps().campaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap()
                                        .withIds(
                                                ids
                                        )
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withOffset(1L)
                                        .withLimit(2L)));
        assertThat("значение LimitedBy верно", getResponse.getLimitedBy(), equalTo(3L));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("453")
    public void useLimitAndNotOwnedCampaign() {
        GetResponse getResponse = api.userSteps.campaignSteps().campaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap()
                                        .withIds(
                                                ids[0],
                                                ids[1],
                                                anotherClientCampID.longValue()
                                        )
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(2L)));
        List<Long> gotids = extractProperty(getResponse.getCampaigns(), CampaignFieldEnum.ID.value());
        assertThat("вернулся правильный набор кампаний", gotids, equalTo(Arrays.asList(ids[0], ids[1])));
    }

}
