package ru.yandex.autotests.direct.api.campaigns.get;

import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;

import com.yandex.direct.api.v5.campaigns.CampaignFieldEnum;
import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import org.joda.time.DateTime;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampOperationsQueueOperation;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampOperationsQueueRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampaignsRecord;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.StatusClarification;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignGetItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.common.Value;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by ginger on 19.08.15.
 * https://st.yandex-team.ru/TESTIRT-6407
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-34568")
@Features(CampaignsFeatures.GET)
@Description("Проверка поля StatusClarification")
public class GetStatusClarificationTest {

    private static final String CLIENT = CampaignsLogins.GET_CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(5).as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private Long id;
    private static int shard;

    //DIRECT-46019

    @BeforeClass
    public static void prepareUser() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        shard = api.userSteps.clientFakeSteps().getUserShard(CLIENT);
    }

    @Before
    public void initTest() {
        id = api.userSteps.campaignSteps().addDefaultTextCampaign();
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("483")
    public void stateArchived() {
        api.userSteps.campaignSteps().campaignsSuspend(id);
        api.userSteps.campaignSteps().campaignsArchive(id);
        CampaignGetItem campaignGetItem =
                api.userSteps.campaignSteps().campaignsGet(
                        id,
                        CampaignFieldEnum.ID, CampaignFieldEnum.STATUS_CLARIFICATION
                );
        assertThat("вернулся правильный StatusClarification", campaignGetItem, beanDifferV5(
                (CampaignGetItem) new CampaignGetItemMap()
                        .withId(id)
                        .withStatusClarification(StatusClarification.CAMPAIGN_ARCHIVED.getStatusClarification())
                        .getBean()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("484")
    public void stateArchivedWaitUnarchive() {
        api.userSteps.campaignSteps().campaignsSuspend(id);
        api.userSteps.campaignSteps().campaignsArchive(id);
        CampOperationsQueueRecord campOperationsQueue = new CampOperationsQueueRecord();
        campOperationsQueue.setCid(id);
        campOperationsQueue.setOperation(CampOperationsQueueOperation.unarc);
        campOperationsQueue.setQueueTime(new Timestamp(DateTime.now().plusSeconds(5).getMillis()));

        api.userSteps.getDirectJooqDbSteps().useShard(shard).campOperationsQueueSteps().saveCampOperationsQueueRecord(campOperationsQueue);
        CampaignGetItem campaignGetItem =
                api.userSteps.campaignSteps().campaignsGet(
                        id,
                        CampaignFieldEnum.ID, CampaignFieldEnum.STATUS_CLARIFICATION
                );
        assertThat("вернулся правильный StatusClarification", campaignGetItem, beanDifferV5(
                (CampaignGetItem) new CampaignGetItemMap()
                        .withId(id)
                        .withStatusClarification(StatusClarification.CAMPAIGN_WAITING_EXTRACTED_FROM_ARCHIVE
                                .getStatusClarification()).getBean()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("485")
    public void noAds() {
        api.userSteps.campaignFakeSteps().setStatusModerate(id, Status.READY);
        CampaignGetItem campaignGetItem =
                api.userSteps.campaignSteps().campaignsGet(
                        id,
                        CampaignFieldEnum.ID, CampaignFieldEnum.STATUS_CLARIFICATION
                );
        assertThat("вернулся правильный StatusClarification", campaignGetItem, beanDifferV5(
                (CampaignGetItem) new CampaignGetItemMap()
                        .withId(id)
                        .withStatusClarification(StatusClarification.NO_ADS.getStatusClarification()).getBean()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("480")
    public void startDate() {
        api.userSteps.campaignFakeSteps().setCampaignSum(
                id, MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().floatValue());
        api.userSteps.campaignFakeSteps().setStatusShow(id, Value.YES);
        api.userSteps.campaignFakeSteps().setStatusModerate(id, Status.YES);
        api.userSteps.campaignFakeSteps().setStatusPostModerate(id, Status.NEW);
        String startDate = DateTime.now().plusDays(1).toLocalDate().toString();
        api.userSteps.campaignSteps().campaignsUpdate(new UpdateRequestMap()
                .withCampaigns(new CampaignUpdateItemMap()
                        .withId(id)
                        .withStartDate(startDate)));
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(id);
        Long adId = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        api.userSteps.bannersFakeSteps().setStatusActive(adId, Value.YES);
        api.userSteps.bannersFakeSteps().setStatusShow(adId, Value.YES);
        CampaignGetItem campaignGetItem =
                api.userSteps.campaignSteps().campaignsGet(
                        id,
                        CampaignFieldEnum.ID, CampaignFieldEnum.STATUS_CLARIFICATION
                );
        String startDateInStatus = DateTime.now().plusDays(1).toLocalDate().toString("dd.MM.yyyy");
        assertThat("вернулся правильный StatusClarification", campaignGetItem, beanDifferV5(
                (CampaignGetItem) new CampaignGetItemMap()
                        .withId(id)
                        .withStatusClarification(
                                String.format(StatusClarification.START.getStatusClarification(), startDateInStatus))
                        .getBean()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("486")
    public void campaignFinished() throws ParseException {
        api.userSteps.campaignFakeSteps().setCampaignSum(
                id, MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().floatValue());
        api.userSteps.campaignFakeSteps().setStatusShow(id, Value.YES);
        api.userSteps.campaignFakeSteps().setStatusModerate(id, Status.YES);
        api.userSteps.campaignFakeSteps().setStatusPostModerate(id, Status.ACCEPTED);
        api.userSteps.campaignFakeSteps().setStatusActive(id, Status.YES);
        api.userSteps.campaignFakeSteps().setBSSynced(id, true);
        String startDate = DateTime.now().minusDays(2).toLocalDate().toString();
        String finishDate = DateTime.now().minusDays(1).toLocalDate().toString();

        int shard = api.userSteps.clientFakeSteps().getUserShard(CLIENT);
        CampaignsRecord campaigns = api.userSteps.getDirectJooqDbSteps().useShard(shard).campaignsSteps().getCampaignById(id);
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        campaigns.setStartTime(new java.sql.Date(formatter.parse(startDate).getTime()));
        campaigns.setFinishTime(new java.sql.Date(formatter.parse(finishDate).getTime()));
        api.userSteps.getDirectJooqDbSteps().useShard(shard).campaignsSteps().updateCampaigns(campaigns);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(id);
        Long adId = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        api.userSteps.bannersFakeSteps().setStatusActive(adId, Value.YES);
        api.userSteps.bannersFakeSteps().setStatusShow(adId, Value.YES);
        CampaignGetItem campaignGetItem =
                api.userSteps.campaignSteps().campaignsGet(
                        id,
                        CampaignFieldEnum.ID, CampaignFieldEnum.STATUS_CLARIFICATION
                );
        String finishDateInStatus = DateTime.now().minusDays(1).toLocalDate().toString("dd.MM.yyyy");
        assertThat("вернулся правильный StatusClarification", campaignGetItem, beanDifferV5(
                (CampaignGetItem) new CampaignGetItemMap()
                        .withId(id)
                        .withStatusClarification(
                                String.format(StatusClarification.CAMPAIGN_ENDED.getStatusClarification(),
                                        finishDateInStatus)).getBean()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("487")
    public void moneyIsOver() {
        api.userSteps.campaignFakeSteps().setCampaignSum(
                id, MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().floatValue());
        api.userSteps.campaignFakeSteps().setSumSpent(
                id, MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().floatValue());
        api.userSteps.campaignFakeSteps().setStatusModerate(id, Status.YES);
        api.userSteps.campaignFakeSteps().setRandomOrderID(id);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(id);
        api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        CampaignGetItem campaignGetItem =
                api.userSteps.campaignSteps().campaignsGet(
                        id,
                        CampaignFieldEnum.ID, CampaignFieldEnum.STATUS_CLARIFICATION
                );
        assertThat("вернулся правильный StatusClarification", campaignGetItem, beanDifferV5(
                (CampaignGetItem) new CampaignGetItemMap()
                        .withId(id)
                        .withStatusClarification(StatusClarification.FUNDS_HAVE_RUN_OUT.getStatusClarification())
                        .getBean()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("481")
    public void acceptedByModerator() {
        api.userSteps.campaignFakeSteps().setStatusModerate(id, Status.YES);
        api.userSteps.campaignFakeSteps().setStatusPostModerate(id, Status.ACCEPTED);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(id);
        api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        CampaignGetItem campaignGetItem =
                api.userSteps.campaignSteps().campaignsGet(
                        id,
                        CampaignFieldEnum.ID, CampaignFieldEnum.STATUS_CLARIFICATION
                );
        assertThat("вернулся правильный StatusClarification", campaignGetItem, beanDifferV5(
                (CampaignGetItem) new CampaignGetItemMap()
                        .withId(id)
                        .withStatusClarification(StatusClarification.ACCEPTED_BY_MODERATOR.getStatusClarification())
                        .getBean()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("488")
    public void rejectedByModerator() {
        api.userSteps.campaignFakeSteps().setStatusModerate(id, Status.NO);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(id);
        api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        CampaignGetItem campaignGetItem =
                api.userSteps.campaignSteps().campaignsGet(
                        id,
                        CampaignFieldEnum.ID, CampaignFieldEnum.STATUS_CLARIFICATION
                );
        assertThat("вернулся правильный StatusClarification", campaignGetItem, beanDifferV5(
                (CampaignGetItem) new CampaignGetItemMap()
                        .withId(id)
                        .withStatusClarification(StatusClarification.REJECTED_BY_MODERATOR.getStatusClarification())
                        .getBean()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("482")
    public void awaitingModeartion() {
        api.userSteps.campaignFakeSteps().setStatusModerate(id, Status.READY);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(id);
        api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        CampaignGetItem campaignGetItem =
                api.userSteps.campaignSteps().campaignsGet(
                        id,
                        CampaignFieldEnum.ID, CampaignFieldEnum.STATUS_CLARIFICATION
                );
        assertThat("вернулся правильный StatusClarification", campaignGetItem, beanDifferV5(
                (CampaignGetItem) new CampaignGetItemMap()
                        .withId(id)
                        .withStatusClarification(
                                StatusClarification.AWAITING_MODERATION_WITHOUT_DOT.getStatusClarification())
                        .getBean()));
    }
}

