package ru.yandex.autotests.direct.api.campaigns.get;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import com.yandex.direct.api.v5.campaigns.CampaignFieldEnum;
import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import com.yandex.direct.api.v5.campaigns.CampaignStateEnum;
import com.yandex.direct.api.v5.campaigns.CampaignStateGetEnum;
import com.yandex.direct.api.v5.campaigns.CampaignStatusPaymentEnum;
import com.yandex.direct.api.v5.campaigns.CampaignStatusSelectionEnum;
import com.yandex.direct.api.v5.campaigns.GetResponse;
import com.yandex.direct.api.v5.general.StatusEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampaignsRecord;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.common.api45mng.APIPort_PortType;
import ru.yandex.autotests.directapi.common.api45mng.CreateNewSubclientResponse;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.enums.AdGroupType;
import ru.yandex.autotests.directapi.enums.BannerType;
import ru.yandex.autotests.directapi.enums.CampaignType;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignGetItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanDiffer;

/**
 * Created by pavryabov on 13.09.15.
 * https://st.yandex-team.ru/TESTIRT-6407
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-34568")
@Features(CampaignsFeatures.GET)
@Description("Проверка фильтрации без Ids")
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class GetWithoutIdsTest {

    private static final String manager = CampaignsLogins.MANAGER_GEO;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private Long draftId;
    private Long activeId;
    private String login;

    @Parameterized.Parameter(0)
    public CampaignType campaignType;

    @Parameterized.Parameter(1)
    public AdGroupType adGroupType;

    @Parameterized.Parameter(2)
    public BannerType bannerType;

    @Parameterized.Parameters(name = "campaignType = {0}, adGroupType = {1}, bannerType = {2}")
    public static Collection testData() {
        Object[][] data = new Object[][]{
                {CampaignType.TEXT, AdGroupType.BASE, BannerType.TEXT},
                {CampaignType.MOBILE_CONTENT, AdGroupType.MOBILE_CONTENT, BannerType.MOBILE_CONTENT},
                {CampaignType.DYNAMIC, AdGroupType.DYNAMIC, BannerType.DYNAMIC},
        };
        return Arrays.asList(data);
    }

    @Before
    public void prepareData() {
        api.as(manager).wsdl(APIPort_PortType.class);
        CreateNewSubclientResponse createNewSubclientResponse =
                api.userSteps.clientSteps().createServicedClient("subClt", manager);
        login = createNewSubclientResponse.getLogin();
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(manager);
        draftId = api.userSteps.campaignSteps().addDefaultCampaign(login, campaignType);
        activeId = api.userSteps.campaignSteps().addDefaultCampaign(login, campaignType);
        api.userSteps.deleteAccountByLogin(login);


        int shard = api.userSteps.clientFakeSteps().getUserShard(login);
        CampaignsRecord campaignRecord = api.userSteps.getDirectJooqDbSteps()
                .useShard(shard).campaignsSteps().getCampaignById(draftId);
        campaignRecord.setWalletCid(0L);
        api.userSteps.getDirectJooqDbSteps()
                .useShard(shard).campaignsSteps().updateCampaigns(campaignRecord);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(activeId, login, adGroupType);
        Long adId = api.userSteps.adsSteps().addDefaultAd(adGroupId, login, bannerType);
        api.userSteps.campaignFakeSteps().setStatusModerate(activeId, Status.YES);
        api.userSteps.bannersFakeSteps().setStatusModerate(adId, Status.YES);
        api.userSteps.campaignFakeSteps().setStatusActive(activeId, Status.YES);
        //делаем одну удаленную кампанию
        Long deletedId = api.userSteps.campaignSteps().addDefaultCampaign(login, campaignType);
        api.userSteps.campaignSteps().campaignsDelete(login, deletedId);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("525")
    public void getWithEmptySelectionCriteria() {
        GetResponse getResponse = api.userSteps.campaignSteps().campaignsGet(new GetRequestMap()
                        .withSelectionCriteria(new CampaignsSelectionCriteriaMap())
                        .withFieldNames(
                                CampaignFieldEnum.ID,
                                CampaignFieldEnum.STATE,
                                CampaignFieldEnum.STATUS,
                                CampaignFieldEnum.STATUS_PAYMENT),
                login);
        assertThat("вернулся правильный ответ", getResponse.getCampaigns(), beanDiffer(
                Arrays.asList((CampaignGetItem) new CampaignGetItemMap()
                                .withId(draftId)
                                .withState(CampaignStateGetEnum.OFF)
                                .withStatus(StatusEnum.DRAFT)
                                .withStatusPayment(CampaignStatusPaymentEnum.DISALLOWED).getBean(),
                        (CampaignGetItem) new CampaignGetItemMap()
                                .withId(activeId)
                                .withState(CampaignStateGetEnum.ON)
                                .withStatus(StatusEnum.ACCEPTED)
                                .withStatusPayment(CampaignStatusPaymentEnum.ALLOWED).getBean())));
    }

    @Test
    public void getWithoutSelectionCriteria() {
        GetResponse getResponse = api.userSteps.campaignSteps().campaignsGet(
                new GetRequestMap().withFieldNames(CampaignFieldEnum.ID),
                login);
        List<CampaignGetItem> expectedCampaigns = Arrays.asList(
                (CampaignGetItem) new CampaignGetItemMap().withId(draftId).getBean(),
                (CampaignGetItem) new CampaignGetItemMap().withId(activeId).getBean());
        assertThat("вернулся правильный ответ", getResponse.getCampaigns(),
                beanDiffer(expectedCampaigns));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("526")
    public void getStateWithoutIds() {
        GetResponse getResponse = api.userSteps.campaignSteps().campaignsGet(new GetRequestMap()
                        .withSelectionCriteria(new CampaignsSelectionCriteriaMap()
                                .withStates(CampaignStateEnum.ON))
                        .withFieldNames(CampaignFieldEnum.ID),
                login);
        assertThat("вернулся правильный ответ", getResponse.getCampaigns(), beanDiffer(
                Arrays.asList((CampaignGetItem) new CampaignGetItemMap()
                        .withId(activeId).getBean())));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("527")
    public void getStatusWithoutIds() {
        GetResponse getResponse = api.userSteps.campaignSteps().campaignsGet(new GetRequestMap()
                        .withSelectionCriteria(new CampaignsSelectionCriteriaMap()
                                .withStatuses(CampaignStatusSelectionEnum.DRAFT))
                        .withFieldNames(CampaignFieldEnum.ID),
                login);
        assertThat("вернулся правильный ответ", getResponse.getCampaigns(), beanDiffer(
                Arrays.asList((CampaignGetItem) new CampaignGetItemMap()
                        .withId(draftId).getBean())));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("528")
    public void getStatusPaymentWithoutIds() {
        GetResponse getResponse = api.userSteps.campaignSteps().campaignsGet(new GetRequestMap()
                        .withSelectionCriteria(new CampaignsSelectionCriteriaMap()
                                .withStatusesPayment(CampaignStatusPaymentEnum.ALLOWED))
                        .withFieldNames(CampaignFieldEnum.ID),
                login);
        assertThat("вернулся правильный ответ", getResponse.getCampaigns(), beanDiffer(
                Arrays.asList((CampaignGetItem) new CampaignGetItemMap()
                        .withId(activeId).getBean())));
    }
}
