package ru.yandex.autotests.direct.api.campaigns.get.textcampaign;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.yandex.direct.api.v5.campaigns.CampaignFieldEnum;
import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import com.yandex.direct.api.v5.campaigns.GetResponse;
import com.yandex.direct.api.v5.campaigns.TextCampaignFieldEnum;
import com.yandex.direct.api.v5.campaigns.TextCampaignSettingGet;
import com.yandex.direct.api.v5.campaigns.TextCampaignSettingsGetEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import one.util.streamex.StreamEx;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.iterableWithSize;
import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by ginger on 25.08.15.
 * https://st.yandex-team.ru/TESTIRT-6407
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-34568")
@Features(CampaignsFeatures.GET)
@Description("Проверка логики поля Settings")
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class GetSharedAccountAndDailyBudgetSettingsTest {

    private final static String clientWithAccount = CampaignsLogins.GET_PROPERTY_CLIENT;
    private final static String clientWithoutAccount = CampaignsLogins.GET_CLIENT;
    private final static String clientDailyBudgetAllowed = CampaignsLogins.GET_DB_ALLOWED_CLIENT;

    private final static Set<TextCampaignSettingsGetEnum> checkedSettings =
            ImmutableSet.of(
                    TextCampaignSettingsGetEnum.SHARED_ACCOUNT_ENABLED,
                    TextCampaignSettingsGetEnum.DAILY_BUDGET_ALLOWED
            );

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(5);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter
    public String testCaseName;

    @Parameterized.Parameter(1)
    public String client;

    @Parameterized.Parameter(2)
    public YesNoEnum expectedDailyBudgetAllowed;

    @Parameterized.Parameter(3)
    public YesNoEnum expectedSharedAccountEnabled;

    private Long campaignId;

    @BeforeClass
    public static void enableSharedAccount() {
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().enableSharedAccount(clientWithAccount);
    }

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][]{
                //DIRECT-45565
                //DIRECT-56254
                {
                        "getWithDayBudgetAdmitted",
                        clientDailyBudgetAllowed,
                        YesNoEnum.YES,
                        YesNoEnum.NO,
                },
                {
                        "getWithoutSharedAccount",
                        clientWithoutAccount,
                        YesNoEnum.YES,
                        YesNoEnum.NO,
                },
                {
                        "getWithSharedAccount",
                        clientWithAccount,
                        YesNoEnum.YES,
                        YesNoEnum.YES,
                },
        });
    }

    @Before
    public void prepareCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignId = api.as(client).userSteps.campaignSteps().addDefaultTextCampaign();
    }

    @Test
    public void getSharedAccountAndDailyBudgetSettingsTest() {
        GetResponse getResponse = api.as(client).userSteps.campaignSteps().campaignsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new CampaignsSelectionCriteriaMap()
                                        .withIds(campaignId)
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withTextCampaignFieldNames(TextCampaignFieldEnum.SETTINGS));
        List<CampaignGetItem> campaignGetItems = getResponse.getCampaigns();
        assumeThat("вернулась одна кампания", campaignGetItems, iterableWithSize(1));

        Map<TextCampaignSettingsGetEnum, YesNoEnum> settings =
                StreamEx.of(campaignGetItems.get(0).getTextCampaign().getSettings())
                        .mapToEntry(TextCampaignSettingGet::getOption, TextCampaignSettingGet::getValue)
                        .filterKeys(checkedSettings::contains)
                        .toMap();
        Map<TextCampaignSettingsGetEnum, YesNoEnum> expectedSettings = ImmutableMap.of(
                TextCampaignSettingsGetEnum.SHARED_ACCOUNT_ENABLED,
                expectedSharedAccountEnabled,
                TextCampaignSettingsGetEnum.DAILY_BUDGET_ALLOWED,
                expectedDailyBudgetAllowed
        );
        assertThat("вернулись ожидаемые значения Settings", settings,
                beanDifferV5(expectedSettings));
    }
}
