package ru.yandex.autotests.direct.api.campaigns.unarchive;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.campaigns.UnarchiveRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by chicos on 20.08.15.
 */
@Aqua.Test
@Features(CampaignsFeatures.UNARCHIVE)
@Stories(ApiStories.UNITS)
@Description("Списание баллов при разархивировании кампании")
public class UnarchiveCampaignsUnitsTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());
    private static final String client = CampaignsLogins.CLIENT_UNITS_UNARCHIVE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final int UNARCHIVE_COST = 10;
    private static final int ITEM_COST_ON_UNARCHIVE = 5;
    private static final int ITEM_COST_ON_UNARCHIVE_FAULT = 20;
    private static final int GENERAL_FAULT_COST = 50;

    private static final Long INVALID_ID = 123456L;
    private static final Long INVALID_ID_2 = 123457L;

    @Before
    public void clearUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
    }

    @Test
    @Description("Разархивирование одиночной кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("693")
    public void archiveSingleCampaignTest() {
        log.info("Заархивируем кампанию");
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignSteps().campaignsSuspend(campaignID);
        api.userSteps.campaignSteps().campaignsArchive(campaignID);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().campaignsUnarchive(campaignID);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость разархивированиия одной кампании", unitsBefore - unitsAfter,
                equalTo(UNARCHIVE_COST + ITEM_COST_ON_UNARCHIVE));
    }

    @Test
    @Description("Разархивирование нескольких кампаний")
    @ru.yandex.qatools.allure.annotations.TestCaseId("697")
    public void archiveSeveralCampaignsTest() {
        log.info("Заархивируем кампании");
        Long campaignID1 = api.userSteps.campaignSteps().addDefaultTextCampaign();
        Long campaignID2 = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignSteps().campaignsSuspend(campaignID1, campaignID2);
        api.userSteps.campaignSteps().campaignsArchive(campaignID1, campaignID2);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().campaignsUnarchive(campaignID1, campaignID2);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость разархивированиия двух кампаний", unitsBefore - unitsAfter,
                equalTo(UNARCHIVE_COST + ITEM_COST_ON_UNARCHIVE * 2));
    }

    @Test
    @Description("Разархивирование одной невалидной кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("698")
    public void archiveInvalidCampaignTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.UNARCHIVE,
                new UnarchiveRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(INVALID_ID)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при разархивировании кампании", unitsBefore - unitsAfter,
                equalTo(UNARCHIVE_COST + ITEM_COST_ON_UNARCHIVE_FAULT));
    }

    @Test
    @Description("Разархивирование двух невалидных кампаний")
    @ru.yandex.qatools.allure.annotations.TestCaseId("699")
    public void archiveFewInvalidCampaignsTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.UNARCHIVE,
                new UnarchiveRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(INVALID_ID, INVALID_ID_2)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при разархивировании двух кампаний", unitsBefore - unitsAfter,
                equalTo(UNARCHIVE_COST + ITEM_COST_ON_UNARCHIVE_FAULT * 2));
    }

    @Test
    @Description("Разархивирование одной корректной и одной невалидной кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("694")
    public void archiveOneValidAndOneInvalidCampaignsTest() {
        log.info("Заархивируем кампанию");
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignSteps().campaignsSuspend(campaignID);
        api.userSteps.campaignSteps().campaignsArchive(campaignID);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().shouldGetResultOn(
                Action.UNARCHIVE,
                new UnarchiveRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignID, INVALID_ID)),
                ExpectedResult.success(campaignID),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при разархивировании нескольких кампаний",
                unitsBefore - unitsAfter,
                equalTo(UNARCHIVE_COST + ITEM_COST_ON_UNARCHIVE + ITEM_COST_ON_UNARCHIVE_FAULT));
    }

    @Test
    @Description("Невалидный запрос на разархивирование кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("695")
    public void generalErrorOnArchiveCampaignTest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.campaignSteps().shouldGetCampaignErrorOn(
                Action.UNARCHIVE,
                new UnarchiveRequestMap().withSelectionCriteria(null),
                new Api5Error(8000, Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                        "params",
                        capitalize(UnarchiveRequestMap.SELECTION_CRITERIA)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость общей ошибки при разархивировании кампании", unitsBefore - unitsAfter,
                equalTo(GENERAL_FAULT_COST));
    }

    @Test
    @Description("Недостаточное колличество баллов при разархивировании кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("696")
    public void notEnoughUnitsOnArchiveCampaignTest() {
        log.info("Заархивируем кампанию");
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignSteps().campaignsSuspend(campaignID);
        api.userSteps.campaignSteps().campaignsArchive(campaignID);

        log.info("Обнулим колличество доступных пользователю баллов");
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(client, unitsBefore);
        unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assumeThat("доступные баллы пользователя", unitsBefore, equalTo(0));

        api.userSteps.campaignSteps().shouldGetCampaignErrorOn(
                Action.UNARCHIVE,
                new UnarchiveRequestMap().withSelectionCriteria(new IdsCriteriaMap()
                        .withIds(campaignID)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("доступные баллы пользователя после ошибки", unitsAfter, equalTo(0));
    }
}
