package ru.yandex.autotests.direct.api.campaigns.unarchive;

import java.util.Arrays;
import java.util.Collection;

import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsArchived;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusbssynced;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampaignsRecord;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.BannerFakeInfo;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.GroupFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.MobileAppAdAddMap;
import ru.yandex.autotests.directapi.model.common.Value;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by chicos on 25.09.15.
 */
@Aqua.Test
@Features(CampaignsFeatures.UNARCHIVE)
@Description("Проверка статусов в БД при разархивации мобильной кампании")
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
@Issue("DIRECT-49971")
@SuppressWarnings("squid:S1192")
public class UnarchiveMobileCampaignsDBStatusTest {
    private static LogSteps log = LogSteps.getLogger(UnarchiveMobileCampaignsDBStatusTest.class);
    private static final String CLIENT = CampaignsLogins.SINGLE_CLIENT_UNARCHIVE;

    private CampaignsRecord campaignDB;
    private GroupFakeInfo groupInfo;
    private BannerFakeInfo adInfo;
    private static final DateTime lastChange = DateTime.now().minusDays(5);

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @SuppressWarnings("DefaultAnnotationParam")
    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public String trackingUrl;

    @Parameterized.Parameter(2)
    public CampaignsStatusmoderate campaignStatusModerate;

    @Parameterized.Parameters(name = "{0}, trackingUrl = {1}, statusModeration = {2}")
    public static Collection<Object[]> data() {
        return Arrays.asList((Object[][]) new Object[][]{
                {"домен объявления помечен как подозрительный, кампания отправляется на перемодерацию",
                        "http://appmetrika.yandex.ru/?logid={logid}&google_aid={google_aid}",
                        CampaignsStatusmoderate.Ready},
                {"домен объявления не помечен, кампания не отправляется на перемодерацию",
                        "https://clicks.bluetrackmedia.com/?logid={logid}&google_aid={google_aid}",
                        CampaignsStatusmoderate.Yes}
        });
    }

    @Before
    @Description("Подготовим данные для теста")
    public void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);

        Long campaignID = api.userSteps.campaignSteps().addDefaultMobileAppCampaign();
        Long groupID = api.userSteps.adGroupsSteps().addDefaultMobileGroup(campaignID);

        Long adID = api.userSteps.adsSteps().addAd(new AdAddItemMap().withAdGroupId(groupID)
                .withMobileAppAd(new MobileAppAdAddMap().defaultMobileAppAd()
                        .withTrackingUrl(trackingUrl)), null);

        api.userSteps.keywordsSteps().addDefaultKeyword(groupID);

        api.userSteps.bannersFakeSteps().makeBannerActive(adID);
        api.userSteps.campaignFakeSteps().makeCampaignActive(campaignID);
        api.userSteps.campaignFakeSteps().setLastShowTimeAnHourAndMinuteAgo(campaignID);
        api.userSteps.campaignFakeSteps().makeCampaignStopped(campaignID);
        api.userSteps.campaignSteps().campaignsArchive(campaignID);

        log.info("Выставим статусы синхронизации с БК фейково");
        api.userSteps.campaignFakeSteps().setBSSynced(campaignID, true);
        api.userSteps.groupFakeSteps().setGroupFakeStatusBsSynced(groupID, Status.YES);
        api.userSteps.bannersFakeSteps().setStatusBsSynced(adID, Status.YES);

        log.info("Сдвинем время последнего изменения на несколько дней");
        api.userSteps.campaignFakeSteps().setLastChange(campaignID, lastChange.toString());
        api.userSteps.groupFakeSteps().setLastChange(groupID, lastChange.toString());
        api.userSteps.bannersFakeSteps().setLastChange(adID, lastChange.toString());

        log.info("Разархивируем кампанию");
        api.userSteps.campaignSteps().campaignsUnarchive(campaignID);
        campaignDB = api.userSteps.getDirectJooqDbSteps().useShardForLogin(CLIENT).campaignsSteps()
                .getCampaignById(campaignID);
        groupInfo = api.userSteps.groupFakeSteps().getGroupParams(groupID);
        adInfo = api.userSteps.bannersFakeSteps().getBannerParams(adID);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("716")
    public void campaignStatusArchivedTest() {
        assertThat("поле Archived после разархивации мобильной кампании", campaignDB.getArchived(),
                equalTo(CampaignsArchived.No));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("717")
    public void campaignStatusModerateTest() {
        assertThat("поле StatusModerate после разархивации мобильной кампании", campaignDB.getStatusmoderate(),
                equalTo(campaignStatusModerate));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("710")
    public void campaignBsSyncedTest() {
        assertThat("поле StatusBsSynced после разархивации мобильной кампании", campaignDB.getStatusbssynced(),
                equalTo(CampaignsStatusbssynced.No));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("711")
    public void campaignLastChangeTest() {
        LocalDate date = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss.0")
                .parseLocalDate(campaignDB.getLastchange().toString());

        assertThat("время последнего изменения кампании после разархивации мобильной кампании", date,
                equalTo(LocalDate.now()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("712")
    public void groupStatusShowsForecastTest() {
        assertThat("поле StatusShowsForecast после разархивации мобильной кампании", groupInfo.getStatusShowsForecast(),
                equalTo(Status.NEW));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("713")
    public void groupStatusBsSyncedTest() {
        assertThat("поле StatusBsSynced после разархивации мобильной кампании", groupInfo.getStatusBsSynced(),
                equalTo(Status.NO));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("714")
    public void adStatusBsSyncedTest() {
        assertThat("поле StatusBsSynced после разархивации мобильной кампании", adInfo.getStatusBsSynced(),
                equalTo(Value.NO));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("715")
    public void adLastChangeTest() {
        LocalDate date = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss").parseLocalDate(adInfo.getLastChange());

        assertThat("время последнего изменения объявления после разархивации мобильной кампании", date,
                equalTo(LocalDate.now()));
    }
}
