package ru.yandex.autotests.direct.api.campaigns.update.cpmbannercampaign.strategy.base;

import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignNetworkStrategyTypeEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import org.joda.time.DateTime;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CpmBannerCampaignNetworkStrategyMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CpmBannerCampaignSearchStrategyMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CpmBannerCampaignStrategyMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CpmBannerCampaignUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.StrategyCpDecreasedPriceForRepeatedImpressionsMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.StrategyCpMaximumImpressionsMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.StrategyWbDecreasedPriceForRepeatedImpressionsMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.StrategyWbMaximumImpressionsMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.directapi.model.api5.campaigns.CampaignAddItemMap.DATE_PATTERN;

@Aqua.Test
@Features(CampaignsFeatures.UPDATE)
@RunWith(Parameterized.class)
@Description("Сценарии проверки параметра AverageCpm при обновлении кампаний")
public class StrategiesAverageCpmTest {

    private static final String CLIENT = CampaignsLogins.CLIENT;
    private static final Currency CURRENCY = Currency.RUB;
    private static final MoneyCurrency MONEY_CURRENCY = MoneyCurrency.get(CURRENCY);

    private static final Money MIN_PRICE = MONEY_CURRENCY.getMinCpmPrice();
    private static final Money MAX_PRICE = MONEY_CURRENCY.getMaxCpmPrice();

    private static final String START_DATE = DateTime.now().plusDays(1).toString(DATE_PATTERN);
    private static final String END_DATE = START_DATE;

    private static final Long SPEND_LIMIT = MONEY_CURRENCY.getMaxDailyBudgetForPeriod().bidLong().longValue();

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public Long averageCpm;

    @Parameterized.Parameter(value = 2)
    public JavaOrPerlExpectedResult expectedResult;

    @Parameterized.Parameters(name = "test = {0}")
    public static Collection strategies() {
        Object[][] data = new Object[][]{
                {
                        "Нулевое значение",
                        0L,
                        JavaOrPerlExpectedResult
                                .java(ExpectedResult.errors(new Notification(4000,
                                        Api5ErrorDetailsJava.VALUE_MUST_NOT_BE_LESS_WITHOUT_CURRENCY,
                                        "AverageCpm",
                                        MIN_PRICE.stringValue(MoneyFormat.WITHOUT_EXTRA_ZEROS))))
                                .perl(ExpectedResult.errors(new Notification(4000,
                                        Api5ErrorDetails.AVERAGE_CPM_SHOULD_BE_MORE_THAN,
                                        MIN_PRICE.stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                                        MONEY_CURRENCY.getApiDescription())))
                },
                {
                        "Отрицательное значение",
                        -1L,
                        JavaOrPerlExpectedResult
                                .java(ExpectedResult.errors(new Notification(4000,
                                        Api5ErrorDetailsJava.VALUE_MUST_NOT_BE_LESS_WITHOUT_CURRENCY,
                                        "AverageCpm",
                                        MIN_PRICE.stringValue(MoneyFormat.WITHOUT_EXTRA_ZEROS))))
                                .perl(ExpectedResult.errors(new Notification(4000,
                                        Api5ErrorDetails.AVERAGE_CPM_SHOULD_BE_MORE_THAN,
                                        MIN_PRICE.stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                                        MONEY_CURRENCY.getApiDescription())))
                },
                {
                        "Значение меньше минимального",
                        MIN_PRICE.getPrevious().bidLong().longValue(),
                        JavaOrPerlExpectedResult
                                .java(ExpectedResult.errors(new Notification(4000,
                                        Api5ErrorDetailsJava.VALUE_MUST_NOT_BE_LESS_WITHOUT_CURRENCY,
                                        "AverageCpm",
                                        MIN_PRICE.stringValue(MoneyFormat.WITHOUT_EXTRA_ZEROS))))
                                .perl(ExpectedResult.errors(new Notification(4000,
                                        Api5ErrorDetails.AVERAGE_CPM_SHOULD_BE_MORE_THAN,
                                        MIN_PRICE.stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                                        MONEY_CURRENCY.getApiDescription())))
                },
                {
                        "Значение больше максимального",
                        MAX_PRICE.getNext().bidLong().longValue(),
                        JavaOrPerlExpectedResult
                                .java(ExpectedResult.errors(new Notification(4000,
                                        Api5ErrorDetailsJava.VALUE_MUST_NOT_BE_GREATER_THAN,
                                        "AverageCpm",
                                        MAX_PRICE.stringValue(MoneyFormat.WITHOUT_EXTRA_ZEROS))))
                                .perl(ExpectedResult.errors(new Notification(4000,
                                        Api5ErrorDetails.AVERAGE_CPM_SHOULD_BE_LESS_THAN,
                                MAX_PRICE.stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                                        MONEY_CURRENCY.getApiDescription())))
                },
                {
                        "Значение равное минимальному",
                        MIN_PRICE.bidLong().longValue(),
                        JavaOrPerlExpectedResult.both(ExpectedResult.success())
                },
                {
                        "Значение равное максимальному",
                        MAX_PRICE.bidLong().longValue(),
                        JavaOrPerlExpectedResult.both(ExpectedResult.success())
                },
        };
        return Arrays.asList(data);
    }

    private long campaignId;

    @BeforeClass
    @Step("Подготовим данные для теста")
    public static void clearUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
    }

    @Before
    public void createCampaign() {
        campaignId = api.userSteps.campaignSteps().addDefaultCpmBannerCampaign();
    }

    @Test
    public void cpMaximumImpressionsTest() {
        api.userSteps.campaignSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withCampaigns(
                        new CampaignUpdateItemMap()
                                .withId(campaignId)
                                .defaultCampaignUpdateItem()
                                .withCpmBannerCampaign(
                                        new CpmBannerCampaignUpdateItemMap()
                                                .withBiddingStrategy(new CpmBannerCampaignStrategyMap()
                                                        .withSearch(new CpmBannerCampaignSearchStrategyMap().defaultServingOff())
                                                        .withNetwork(
                                                                new CpmBannerCampaignNetworkStrategyMap()
                                                                        .withBiddingStrategyType(
                                                                                CpmBannerCampaignNetworkStrategyTypeEnum.CP_MAXIMUM_IMPRESSIONS
                                                                        )
                                                                        .withCpMaximumImpressions(
                                                                                new StrategyCpMaximumImpressionsMap()
                                                                                        .withStartDate(START_DATE)
                                                                                        .withEndDate(END_DATE)
                                                                                        .withAutoContinue(YesNoEnum.YES)
                                                                                        .withAverageCpm(averageCpm)
                                                                                        .withSpendLimit(SPEND_LIMIT)
                                                                                        .withAutoContinue(YesNoEnum.YES)))))),
                expectedResult);
    }

    @Test
    public void wbMaximumImpressionsTest() {
        api.userSteps.campaignSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withCampaigns(
                        new CampaignUpdateItemMap()
                                .withId(campaignId)
                                .defaultCampaignUpdateItem()
                                .withCpmBannerCampaign(
                                        new CpmBannerCampaignUpdateItemMap()
                                                .withBiddingStrategy(new CpmBannerCampaignStrategyMap()
                                                        .withSearch(new CpmBannerCampaignSearchStrategyMap().defaultServingOff())
                                                        .withNetwork(
                                                                new CpmBannerCampaignNetworkStrategyMap()
                                                                        .withBiddingStrategyType(
                                                                                CpmBannerCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_IMPRESSIONS
                                                                        )
                                                                        .withWbMaximumImpressions(
                                                                                new StrategyWbMaximumImpressionsMap()
                                                                                        .withAverageCpm(averageCpm)
                                                                                        .withSpendLimit(SPEND_LIMIT)))))),
                expectedResult);
    }

    @Test
    public void cpDecreasedPriceForRepeatedImpressionsTest() {
        api.userSteps.campaignSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withCampaigns(
                        new CampaignUpdateItemMap()
                                .withId(campaignId)
                                .defaultCampaignUpdateItem()
                                .withCpmBannerCampaign(
                                        new CpmBannerCampaignUpdateItemMap()
                                                .withBiddingStrategy(new CpmBannerCampaignStrategyMap()
                                                        .withSearch(new CpmBannerCampaignSearchStrategyMap().defaultServingOff())
                                                        .withNetwork(
                                                                new CpmBannerCampaignNetworkStrategyMap()
                                                                        .withBiddingStrategyType(
                                                                                CpmBannerCampaignNetworkStrategyTypeEnum.CP_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS
                                                                        )
                                                                        .withCpDecreasedPriceForRepeatedImpressions(
                                                                                new StrategyCpDecreasedPriceForRepeatedImpressionsMap()
                                                                                        .withStartDate(START_DATE)
                                                                                        .withEndDate(END_DATE)
                                                                                        .withAverageCpm(averageCpm)
                                                                                        .withSpendLimit(SPEND_LIMIT)
                                                                                        .withAutoContinue(YesNoEnum.YES)))))),
                expectedResult);
    }

    @Test
    public void wbDecreasedPriceForRepeatedImpressionsTest() {
        api.userSteps.campaignSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withCampaigns(
                        new CampaignUpdateItemMap()
                                .withId(campaignId)
                                .defaultCampaignUpdateItem()
                                .withCpmBannerCampaign(
                                        new CpmBannerCampaignUpdateItemMap()
                                                .withBiddingStrategy(new CpmBannerCampaignStrategyMap()
                                                        .withSearch(new CpmBannerCampaignSearchStrategyMap().defaultServingOff())
                                                        .withNetwork(
                                                                new CpmBannerCampaignNetworkStrategyMap()
                                                                        .withBiddingStrategyType(
                                                                                CpmBannerCampaignNetworkStrategyTypeEnum.WB_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS
                                                                        )
                                                                        .withWbDecreasedPriceForRepeatedImpressions(
                                                                                new StrategyWbDecreasedPriceForRepeatedImpressionsMap()
                                                                                        .withAverageCpm(averageCpm)
                                                                                        .withSpendLimit(SPEND_LIMIT)))))),
                expectedResult);
    }
}
