package ru.yandex.autotests.direct.api.campaigns.update.cpmbannercampaign.strategy.cp;

import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignNetworkStrategyTypeEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import org.joda.time.DateTime;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CpmBannerCampaignNetworkStrategyMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CpmBannerCampaignSearchStrategyMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CpmBannerCampaignStrategyMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CpmBannerCampaignUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.StrategyCpDecreasedPriceForRepeatedImpressionsMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.StrategyCpMaximumImpressionsMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

@Aqua.Test
@Features(CampaignsFeatures.UPDATE)
@RunWith(Parameterized.class)
@Description("Негативные сценарии проверки параметра SpendLimit при обновлении кампании со стратегиями на фиксированный период")
public class StrategiesCustomPeriodSpendLimitTest {

    private static final String CLIENT = CampaignsLogins.CLIENT;
    private static final Currency CURRENCY = Currency.RUB;
    private static final MoneyCurrency MONEY_CURRENCY = MoneyCurrency.get(CURRENCY);

    private static final long AVERAGE_CPM = MONEY_CURRENCY.getMinCpmPrice().bidLong().longValue();
    private static final Money MIN_LIMIT = MONEY_CURRENCY.getMinDailyBudgetForPeriod();
    private static final Money MAX_LIMIT = MONEY_CURRENCY.getMaxDailyBudgetForPeriod();

    private static final String START_DATE = DateTime.now().plusDays(1).toString(CampaignAddItemMap.DATE_PATTERN);
    private static final String END_DATE = START_DATE;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public Long spendLimit;

    @Parameterized.Parameter(value = 2)
    public JavaOrPerlExpectedResult expectedResult;

    @Parameterized.Parameters(name = "test = {0}")
    public static Collection strategies() {
        Object[][] data = new Object[][]{
                {
                        "Нулевое значение",
                        0L,
                        JavaOrPerlExpectedResult
                                .java(ExpectedResult.errors(new Notification(4000,
                                        Api5ErrorDetailsJava.FIXED_PERIOD_BUDGET_CANNOT_BE_LESS_WITHOUT_CURRENCY,
                                        MIN_LIMIT.stringValue(MoneyFormat.WITHOUT_EXTRA_ZEROS))))
                                .perl(ExpectedResult.errors(new Notification(4000,
                                        Api5ErrorDetails.FIXED_PERIOD_BUDGET_CANNOT_BE_LESS,
                                        MIN_LIMIT.stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                                        MONEY_CURRENCY.getApiDescription())))
                },
                {
                        "Отрицательное значение",
                        -1L,
                        JavaOrPerlExpectedResult
                                .java(ExpectedResult.errors(new Notification(4000,
                                        Api5ErrorDetailsJava.FIXED_PERIOD_BUDGET_CANNOT_BE_LESS_WITHOUT_CURRENCY,
                                        MIN_LIMIT.stringValue(MoneyFormat.WITHOUT_EXTRA_ZEROS))))
                                .perl(ExpectedResult.errors(new Notification(4000,
                                        Api5ErrorDetails.FIXED_PERIOD_BUDGET_CANNOT_BE_LESS,
                                        MIN_LIMIT.stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                                        MONEY_CURRENCY.getApiDescription())))
                },
                {
                        "Значение меньше минимального",
                        MIN_LIMIT.getPrevious().bidLong().longValue(),
                        JavaOrPerlExpectedResult
                                .java(ExpectedResult.errors(new Notification(4000,
                                        Api5ErrorDetailsJava.FIXED_PERIOD_BUDGET_CANNOT_BE_LESS_WITHOUT_CURRENCY,
                                        MIN_LIMIT.stringValue(MoneyFormat.WITHOUT_EXTRA_ZEROS))))
                                .perl(ExpectedResult.errors(new Notification(4000,
                                        Api5ErrorDetails.FIXED_PERIOD_BUDGET_CANNOT_BE_LESS,
                                        MIN_LIMIT.stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                                        MONEY_CURRENCY.getApiDescription())))
                },
                {
                        "Значение больше максимального",
                        MAX_LIMIT.getNext().bidLong().longValue(),
                        JavaOrPerlExpectedResult
                                .java(ExpectedResult.errors(new Notification(4000,
                                        Api5ErrorDetailsJava.FIXED_PERIOD_BUDGET_CANNOT_BE_MORE_WITHOUT_CURRENCY,
                                        MAX_LIMIT.stringValue(MoneyFormat.WITHOUT_EXTRA_ZEROS))))
                                .perl(ExpectedResult.errors(new Notification(4000,
                                        Api5ErrorDetails.FIXED_PERIOD_BUDGET_CANNOT_BE_MORE,
                                        MAX_LIMIT.stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                                        MONEY_CURRENCY.getApiDescription())))
                },
                {
                        "Значение равное минимальному",
                        MIN_LIMIT.bidLong().longValue(),
                        JavaOrPerlExpectedResult.both(ExpectedResult.success())
                },
                {
                        "Значение равное максимальному",
                        MAX_LIMIT.bidLong().longValue(),
                        JavaOrPerlExpectedResult.both(ExpectedResult.success())
                },
        };
        return Arrays.asList(data);
    }

    private Long campaignId;

    @BeforeClass
    @Step("Подготовим данные для теста")
    public static void clearUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
    }

    @Before
    public void createCampaign() {
        campaignId = api.userSteps.campaignSteps().addDefaultCpmBannerCampaign();
    }

    @Test
    public void cpMaximumImpressionsSpendLimitNegativeTest() {
        api.userSteps.campaignSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withCampaigns(new CampaignUpdateItemMap()
                        .withId(campaignId)
                        .withCpmBannerCampaign(new CpmBannerCampaignUpdateItemMap()
                                .withBiddingStrategy(new CpmBannerCampaignStrategyMap()
                                        .withSearch(new CpmBannerCampaignSearchStrategyMap().defaultServingOff())
                                        .withNetwork(
                                                new CpmBannerCampaignNetworkStrategyMap()
                                                        .withBiddingStrategyType(
                                                            CpmBannerCampaignNetworkStrategyTypeEnum.CP_MAXIMUM_IMPRESSIONS
                                                        )
                                                .withCpMaximumImpressions(
                                                        new StrategyCpMaximumImpressionsMap()
                                                                .withStartDate(START_DATE)
                                                                .withEndDate(END_DATE)
                                                                .withAverageCpm(AVERAGE_CPM)
                                                                .withSpendLimit(spendLimit)
                                                                .withAutoContinue(YesNoEnum.YES)))))),
                expectedResult);
    }

    @Test
    public void cpDecreasedPriceForRepeatedImpressionsSpendLimitNegativeTest() {
        api.userSteps.campaignSteps().shouldGetResultOnUpdate(
                new UpdateRequestMap().withCampaigns(new CampaignUpdateItemMap()
                        .withId(campaignId)
                        .withCpmBannerCampaign(new CpmBannerCampaignUpdateItemMap()
                                .withBiddingStrategy(new CpmBannerCampaignStrategyMap()
                                        .withSearch(new CpmBannerCampaignSearchStrategyMap().defaultServingOff())
                                        .withNetwork(
                                                new CpmBannerCampaignNetworkStrategyMap()
                                                        .withBiddingStrategyType(
                                                                CpmBannerCampaignNetworkStrategyTypeEnum.CP_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS
                                                        )
                                                        .withCpDecreasedPriceForRepeatedImpressions(
                                                                new StrategyCpDecreasedPriceForRepeatedImpressionsMap()
                                                                        .withStartDate(START_DATE)
                                                                        .withEndDate(END_DATE)
                                                                        .withAverageCpm(AVERAGE_CPM)
                                                                        .withSpendLimit(spendLimit)
                                                                        .withAutoContinue(YesNoEnum.YES)))))),
                expectedResult);
    }
}
