package ru.yandex.autotests.direct.api.campaigns.update.dailybudget;

import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.campaigns.DailyBudgetModeEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.DailyBudgetMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult.both;

/**
 * Created by pavryabov on 11.09.15.
 * https://st.yandex-team.ru/TESTIRT-6809
 */
@Aqua.Test
@Features(CampaignsFeatures.UPDATE)
@Description("Проверка лимита на обновление дневного бюджета")
@Issue("https://st.yandex-team.ru/DIRECT-34570")
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class UpdateDailyBudgetLimitTest {

    private static String client = CampaignsLogins.UPDATE_CLIENT;

    private static final Long MAX_DAILY_BUDGET = 1_000_000_000L;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(5).as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long id;

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public DailyBudgetMap dailyBudgetForCreate;

    @Parameterized.Parameter(2)
    public DailyBudgetMap dailyBudgetForUpdate;

    @Parameterized.Parameter(3)
    public int numberOfChangeBeforeUpdate;

    @Parameterized.Parameter(4)
    public JavaOrPerlExpectedResult expectedResult;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        Money minDayBudgetAmount = MoneyCurrency.get(Currency.RUB).getMinDayBudgetAmount();
        Money maxDayBudgetAmount = Money.valueOf(MAX_DAILY_BUDGET);
        Long minRubAmount = minDayBudgetAmount.bidLong().longValue();
        return Arrays.asList(new Object[][]{
                {"Update amount third time",
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.STANDARD),
                        new DailyBudgetMap().withAmount(minRubAmount * 2).withMode(DailyBudgetModeEnum.STANDARD),
                        2, both(ExpectedResult.success())},
                {"Update amount fourth time",
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.STANDARD),
                        new DailyBudgetMap().withAmount(minRubAmount * 2).withMode(DailyBudgetModeEnum.STANDARD),
                        3, both(ExpectedResult.errors(
                        new Notification(8305, Api5ErrorDetails.CHANGE_DAY_BUDGET_NOT_MORE_THAN_3_TIMES)))},
                {"Update mode fourth time",
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.DISTRIBUTED),
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.STANDARD),
                        3, both(ExpectedResult.success())},
                {"Reset DailyBudget after three updates",
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.STANDARD),
                        null,
                        3, both(ExpectedResult.success())},
                {"Update amount to a value less than minimum",
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.STANDARD),
                        new DailyBudgetMap()
                                .withAmount(minDayBudgetAmount.getPrevious().bidLong().longValue())
                                .withMode(DailyBudgetModeEnum.STANDARD),
                        2, JavaOrPerlExpectedResult
                        .java(ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetailsJava.FIELD_MUST_BE_IN_RANGE_FROM_TO,
                                "DailyBudget", minDayBudgetAmount.longValue(), MAX_DAILY_BUDGET)))
                        .perl(ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.DAILY_BUDGET_MINIMUM_WITH_CURRENCY,
                                minDayBudgetAmount.stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                                MoneyCurrency.get(Currency.RUB).getAPIAbbreviation("en"))))
                },
                {"Update amount to a zero value",
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.STANDARD),
                        new DailyBudgetMap().withAmount(0L).withMode(DailyBudgetModeEnum.STANDARD),
                        2, both(ExpectedResult.errors(new Notification(5005,
                        Api5ErrorDetails.FIELD_MUST_BE_POSITIVE_INTEGER, DailyBudgetMap.AMOUNT_IN_ERROR)))
                },
                {"Update amount to a value more than maximum",
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.STANDARD),
                        new DailyBudgetMap()
                                .withAmount(maxDayBudgetAmount.getNext().bidLong().longValue())
                                .withMode(DailyBudgetModeEnum.STANDARD),
                        2, JavaOrPerlExpectedResult
                        .java(ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetailsJava.FIELD_MUST_BE_IN_RANGE_FROM_TO,
                                "DailyBudget", minDayBudgetAmount.longValue(), MAX_DAILY_BUDGET)))
                        .perl(ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetails.DAILY_BUDGET_MAXIMUM_WITH_CURRENCY,
                                maxDayBudgetAmount.stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                                MoneyCurrency.get(Currency.RUB).getAPIAbbreviation("en"))))
                },
        });
    }

    @BeforeClass
    public static void prepareClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
    }

    @Before
    public void createCampaign() {
        id = api.userSteps.campaignSteps().addCampaign(new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withDailyBudget(dailyBudgetForCreate)
                .withDefaultTextCampaign());
        api.userSteps.campaignFakeSteps().setDayBudgetChangeCount(id.intValue(), numberOfChangeBeforeUpdate);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("941")
    public void updateDailyBudget() {
        //DIRECT-46261
        api.userSteps.campaignSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withCampaigns(new CampaignUpdateItemMap()
                                .withId(id)
                                .withDailyBudget(dailyBudgetForUpdate)
                        ),
                expectedResult);
    }
}
