package ru.yandex.autotests.direct.api.campaigns.update.dailybudget;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.campaigns.CampaignFieldEnum;
import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import com.yandex.direct.api.v5.campaigns.DailyBudgetModeEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignGetItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.DailyBudgetMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.core.IsEqual.equalTo;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.tables.Strategies.STRATEGIES;
import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by pavryabov on 11.09.15.
 * https://st.yandex-team.ru/TESTIRT-6809
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-34570")
@Features(CampaignsFeatures.UPDATE)
@Description("Проверка обновления дневного бюджета")
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class UpdateDailyBudgetTest {

    private static String client = CampaignsLogins.UPDATE_CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(5).as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long id;

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public DailyBudgetMap dailyBudgetForCreate;

    @Parameterized.Parameter(2)
    public DailyBudgetMap dailyBudgetForUpdate;

    @Parameterized.Parameter(3)
    public ProtocolType protocolType;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> data() {
        Long minRubAmount = MoneyCurrency.get(Currency.RUB).getMinDayBudgetAmount().bidLong().longValue();
        return Arrays.asList(new Object[][]{
                {"Mode STANDARD -> DISTRIBUTED",
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.STANDARD),
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.DISTRIBUTED),
                        ProtocolType.JSON},
                {"Mode DISTRIBUTED -> STANDARD",
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.DISTRIBUTED),
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.STANDARD),
                        ProtocolType.JSON},
                {"null -> STANDARD",
                        null,
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.DISTRIBUTED),
                        ProtocolType.JSON},
                {"null -> DISTRIBUTED",
                        null,
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.DISTRIBUTED),
                        ProtocolType.JSON},
                {"Update amount",
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.STANDARD),
                        new DailyBudgetMap().withAmount(minRubAmount * 2).withMode(DailyBudgetModeEnum.STANDARD),
                        ProtocolType.JSON},
                {"Update amount and mode",
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.STANDARD),
                        new DailyBudgetMap().withAmount(minRubAmount * 2).withMode(DailyBudgetModeEnum.DISTRIBUTED),
                        ProtocolType.JSON},
                {"Reset DailyBudget JSON",
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.STANDARD),
                        null, ProtocolType.JSON},
                {"Reset DailyBudget SOAP",
                        new DailyBudgetMap().withAmount(minRubAmount).withMode(DailyBudgetModeEnum.STANDARD),
                        null, ProtocolType.SOAP},
        });
    }

    @BeforeClass
    public static void prepareClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
    }

    @Before
    public void createCampaign() {
        id = api.userSteps.campaignSteps().addCampaign(new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withDailyBudget(dailyBudgetForCreate)
                .withDefaultTextCampaign());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("942")
    public void updateDailyBudget() {
        //DIRECT-46245
        //DIRECT-46219
        api.protocol(protocolType);
        api.userSteps.campaignSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withCampaigns(new CampaignUpdateItemMap()
                                .withId(id)
                                .withDailyBudget(dailyBudgetForUpdate)
                        ),
                ExpectedResult.success());
        CampaignGetItem campaignGetItem =
                api.userSteps.campaignSteps().campaignsGet(id, CampaignFieldEnum.DAILY_BUDGET);
        Long strategyId = (Long) api.userSteps.getDirectJooqDbSteps().useShardForLogin(client)
                .campaignsSteps().getCampaignOption(id, CAMPAIGNS.STRATEGY_ID);

        Long strategyDayBudgetAmountInMicros = getStrategyDayBudgetAmountInMicros(strategyId);

        Long newDayBudgetAmount = dailyBudgetForUpdate != null
                ? (Long) dailyBudgetForUpdate.getOrDefault(DailyBudgetMap.AMOUNT, BigDecimal.ZERO)
                : 0;

        assertThat("на стратегии дневной бюджет правильно обновился", strategyDayBudgetAmountInMicros, equalTo(newDayBudgetAmount));
        assertThat("кампания правильно обновилась", campaignGetItem,
                beanDifferV5((CampaignGetItem)
                        new CampaignGetItemMap()
                                .withDailyBudget(dailyBudgetForUpdate).getBean()));
    }

    private Long getStrategyDayBudgetAmountInMicros(Long strategyId) {
        BigDecimal strategyDayBudgetAmount = api.userSteps.getDirectJooqDbSteps().strategiesSteps().getStrategiesRecord(strategyId)
                        .get(STRATEGIES.DAY_BUDGET);
        Long strategyDayBudgetAmountInMicros = strategyDayBudgetAmount
                .multiply(BigDecimal.valueOf(1_000_000))
                .longValue();
        return strategyDayBudgetAmountInMicros;
    }
}
