package ru.yandex.autotests.direct.api.campaigns.update.textcampaign;

import java.sql.Timestamp;
import java.time.LocalDateTime;

import com.yandex.direct.api.v5.general.OperationEnum;
import org.hamcrest.Matchers;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsStatusbssynced;
import ru.yandex.autotests.direct.db.steps.CampaignsSteps;
import ru.yandex.autotests.direct.db.steps.FeaturesSteps;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.PriorityGoalsArrayMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.PriorityGoalsItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.PriorityGoalsUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.PriorityGoalsUpdateSettingMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.Matchers.greaterThan;
import static ru.yandex.autotests.direct.api.campaigns.CampaignCommonMatchers.getMeaningfulGoalsJsonMatcher;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by liosha on 14.01.2019.
 * https://st.yandex-team.ru/DIRECT-76294
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-76294")
@Features(CampaignsFeatures.UPDATE)
@Description("Проверка обновления поля PriorityGoals. Позитивные сценарии")
public class UpdatePriorityGoalsDBValuesTest {

    private static String client = CampaignsLogins.UPDATE_CLIENT;
    private static final Integer COUNTER_ID = 34981090;
    private static final Long GOAL_ID = 17108070L;
    // цель CampaignsLogins.CLIENT, недоступная UPDATE_CLIENT
    private static final Integer ALIEN_COUNTER_ID = 32265784;
    private static final Long ALIEN_GOAL_ID = 13088934L;
    private static int shard;
    private static CampaignsSteps campaignJooqSteps;
    private static FeaturesSteps featuresSteps;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(5).as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @BeforeClass
    public static void createCampaign() {
        shard = api.userSteps.clientFakeSteps().getUserShard(client);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        campaignJooqSteps = api.userSteps.getDirectJooqDbSteps().useShard(shard).campaignsSteps();
        featuresSteps = api.userSteps.getDirectJooqDbSteps().useShard(shard).featuresSteps();
    }

    @Test
    public void updateWithEngagedSessionGoal() {
        Long cid = api.userSteps.campaignSteps().addDefaultTextCampaign();

        api.userSteps.campaignSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withCampaigns(new CampaignUpdateItemMap()
                                .withId(cid)
                                .withTextCampaign(new TextCampaignUpdateItemMap()
                                        .withPriorityGoals(new PriorityGoalsUpdateItemMap()
                                                .withEngagedSessionGoalId()
                                                .withValue(1_000_000L)
                                                .withOperation(OperationEnum.SET)
                                        )
                                )
                        ),
                ExpectedResult.success());

        assertThat("meaningful_goals содержит правильный json",
                campaignJooqSteps.getCampOptionsById(cid).getMeaningfulGoals(),
                getMeaningfulGoalsJsonMatcher(PriorityGoalsItemMap.ENGAGED_SESSION_GOAL_ID, 1L));
    }

    @Test
    public void updateWithOldCounterGoal() {
        Long cid = api.userSteps.campaignSteps().addCampaign(new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withTextCampaign(new TextCampaignAddItemMap()
                        .withBiddingStrategy(new TextCampaignStrategyAddMap().defaultTextCampaignStrategy())
                        .withCounterIds(COUNTER_ID)));

        updateAndCheckWithCounterGoal(cid, null, GOAL_ID);
    }

    @Test
    public void updateWithNewCounterGoal() {
        Long cid = api.userSteps.campaignSteps().addDefaultTextCampaign();

        updateAndCheckWithCounterGoal(cid, COUNTER_ID, GOAL_ID);
    }

    @Test
    public void updateWithOldInaccessibleCounterGoal() {
        Long clientId = Long.parseLong(User.get(client).getClientID());
        featuresSteps.setClientFeature(clientId, "direct_unavailable_goals_allowed", true);

        Long cid = api.userSteps.campaignSteps().addCampaign(new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withTextCampaign(new TextCampaignAddItemMap()
                        .withBiddingStrategy(new TextCampaignStrategyAddMap().defaultTextCampaignStrategy())
                        .withCounterIds(ALIEN_COUNTER_ID)));

        updateAndCheckWithCounterGoal(cid, null, ALIEN_GOAL_ID);
    }

    @Test
    public void updateWithNewInaccessibleCounterGoal() {
        Long clientId = Long.parseLong(User.get(client).getClientID());
        featuresSteps.setClientFeature(clientId, "direct_unavailable_goals_allowed", true);

        Long cid = api.userSteps.campaignSteps().addDefaultTextCampaign();

        updateAndCheckWithCounterGoal(cid, ALIEN_COUNTER_ID, ALIEN_GOAL_ID);
    }

    private void updateAndCheckWithCounterGoal(Long cid, Integer counterId, long goalId) {
        TextCampaignUpdateItemMap campaign = new TextCampaignUpdateItemMap()
                .withPriorityGoals(new PriorityGoalsUpdateItemMap()
                        .withGoalId(goalId)
                        .withValue(1_000_000L)
                        .withOperation(OperationEnum.SET)
                );
        if (counterId != null) {
            campaign.withCounterIds(counterId);
        }
        api.userSteps.campaignSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withCampaigns(new CampaignUpdateItemMap()
                                .withId(cid)
                                .withTextCampaign(campaign)
                        ),
                ExpectedResult.success());

        assertThat("meaningful_goals содержит правильный json",
                campaignJooqSteps.getCampOptionsById(cid).getMeaningfulGoals(),
                getMeaningfulGoalsJsonMatcher(goalId, 1L));
    }

    @Test
    public void updateWithEmptyGoals() {
        Long cid = api.userSteps.campaignSteps().addCampaign(new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withTextCampaign(new TextCampaignAddItemMap()
                        .withBiddingStrategy(new TextCampaignStrategyAddMap().defaultTextCampaignStrategy())
                        .withPriorityGoals(new PriorityGoalsArrayMap()
                                .withItems(new PriorityGoalsItemMap()
                                        .withEngagedSessionGoalId()
                                        .withValue(1_000_000L))))
        );

        api.userSteps.campaignSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withCampaigns(new CampaignUpdateItemMap()
                                .withId(cid)
                                .withTextCampaign(new TextCampaignUpdateItemMap()
                                        .withPriorityGoals((PriorityGoalsUpdateSettingMap) null)
                                )
                        ),
                ExpectedResult.success());

        assertThat("meaningful_goals содержит правильный json",
                campaignJooqSteps.getCampOptionsById(cid).getMeaningfulGoals(),
                nullValue());
    }

    @Test
    public void updateWithChangedValue() {
        Long cid = api.userSteps.campaignSteps().addCampaign(new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withTextCampaign(new TextCampaignAddItemMap()
                        .withBiddingStrategy(new TextCampaignStrategyAddMap().defaultTextCampaignStrategy())
                        .withPriorityGoals(new PriorityGoalsArrayMap()
                                .withItems(new PriorityGoalsItemMap()
                                        .withEngagedSessionGoalId()
                                        .withValue(1_000_000L))))
        );

        api.userSteps.campaignSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withCampaigns(new CampaignUpdateItemMap()
                                .withId(cid)
                                .withTextCampaign(new TextCampaignUpdateItemMap()
                                        .withPriorityGoals(new PriorityGoalsUpdateItemMap()
                                                .withEngagedSessionGoalId()
                                                .withValue(2_000_000L)
                                                .withOperation(OperationEnum.SET)
                                        )
                                )
                        ),
                ExpectedResult.success());

        assertThat("meaningful_goals содержит правильный json",
                campaignJooqSteps.getCampOptionsById(cid).getMeaningfulGoals(),
                getMeaningfulGoalsJsonMatcher(PriorityGoalsItemMap.ENGAGED_SESSION_GOAL_ID, 2L));

    }

    @Test
    public void updateCheckStatusBsSynced() {
        Long cid = api.userSteps.campaignSteps().addDefaultTextCampaign();

        campaignJooqSteps.setCampaignsStatusBsSynced(cid, CampaignsStatusbssynced.Yes);

        api.userSteps.campaignSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withCampaigns(new CampaignUpdateItemMap()
                                .withId(cid)
                                .withTextCampaign(new TextCampaignUpdateItemMap()
                                        .withPriorityGoals(new PriorityGoalsUpdateItemMap()
                                                .withEngagedSessionGoalId()
                                                .withValue(1_000_000L)
                                                .withOperation(OperationEnum.SET)
                                        )
                                )
                        ),
                ExpectedResult.success());

        assertThat("у кампании сбросился StatusBsSynced",
                campaignJooqSteps.getCampaignById(cid).getStatusbssynced(),
                Matchers.equalTo(CampaignsStatusbssynced.No));
    }

    @Test
    public void updateCheckLastChange() {
        Long cid = api.userSteps.campaignSteps().addDefaultTextCampaign();

        Timestamp lastChange = Timestamp.valueOf(LocalDateTime.now().minusDays(5).withNano(0));
        campaignJooqSteps.setCampaignsLastChange(cid, lastChange);

        api.userSteps.campaignSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withCampaigns(new CampaignUpdateItemMap()
                                .withId(cid)
                                .withTextCampaign(new TextCampaignUpdateItemMap()
                                        .withPriorityGoals(new PriorityGoalsUpdateItemMap()
                                                .withEngagedSessionGoalId()
                                                .withValue(1_000_000L)
                                                .withOperation(OperationEnum.SET)
                                        )
                                )
                        ),
                ExpectedResult.success());

        assertThat("LastChange у кампании изменился",
                campaignJooqSteps.getCampaignById(cid).getLastchange().toLocalDateTime(),
                greaterThan(lastChange.toLocalDateTime()));
    }
}
