package ru.yandex.autotests.direct.api.campaigns.update.textcampaign.strategy.bids;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.campaigns.TextCampaignNetworkStrategyTypeEnum;
import com.yandex.direct.api.v5.general.PriorityEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.direct.api.campaigns.CampaignsLogins;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BidsBaseRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.features.FeatureNameEnum;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.StrategyNetworkDefaultAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignNetworkStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignNetworkStrategyMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignSearchStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignSearchStrategyMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignStrategyMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.TextCampaignUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.UpdateRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by adubinkin on 11.02.19.
 */

@Aqua.Test
@Features(CampaignsFeatures.UPDATE)
@Description("Проверка изменений ставки автотаргетигна на сеть при смене стратегии")
@Issue("https://st.yandex-team.ru/DIRECT-91581")
@RunWith(Parameterized.class)
@Tag(TagDictionary.TRUNK)
public class PriceContextChangesWhenStrategyChangeTest {

    private static final MoneyCurrency CURRENCY = MoneyCurrency.get(Currency.RUB);
    private static final Long MIN_PRICE = CURRENCY.getLongMinPrice().longValue();
    private static final Long DEFAULT_PRICE = CURRENCY.getDefaultPriceConstructorAmount().longValue() * 10000L;
    private static final Long BID1 = MIN_PRICE * 10L;
    private static final Long BID2 = 2L * BID1;
    private static final Long ZERO_PRICE = 0L;
    private static final String LOGIN = CampaignsLogins.SINGLE_CLIENT_ANOTHER;

    private static final TextCampaignStrategyAddMap MANUAL_STRATEGY = new TextCampaignStrategyAddMap()
            .withSearch(new TextCampaignSearchStrategyAddMap().defaultHighestPosition())
            .withNetwork(new TextCampaignNetworkStrategyAddMap().defaultServingOff());
    private static final TextCampaignStrategyAddMap AUTO_STRATEGY = new TextCampaignStrategyAddMap()
            .withSearch(new TextCampaignSearchStrategyAddMap().defaultWbMaximumClicks(CURRENCY.getCurrency()))
            .withNetwork(new TextCampaignNetworkStrategyAddMap()
                    .withBiddingStrategyType(TextCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT)
                    .withNetworkDefault((new StrategyNetworkDefaultAddMap())
                            .defaultStrategyNetworkDefault())
            );

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private static Long campaignId;

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(0)
    public Boolean withKeywords;

    @Parameterized.Parameter(1)
    public BigDecimal expectedPrice;

    @Parameterized.Parameter(2)
    public BigDecimal expectedCtxPrice;

    @Parameterized.Parameter(3)
    public TextCampaignStrategyAddMap initialStrategy;

    @Parameterized.Parameters(
            name = "ctxRmFeature={0}, withKeywords={1}, expectedPrice={2}, expectedCtxPrice={3}, strategy={4}")
    public static Collection data() {
        return Arrays.asList(new Object[][]{
                {false, bidToPrice(MIN_PRICE), bidToPrice(DEFAULT_PRICE), MANUAL_STRATEGY},
                {false, bidToPrice(DEFAULT_PRICE), bidToPrice(DEFAULT_PRICE), AUTO_STRATEGY},
                //ставка на сети вычисляется аналогично средневзвешенной на поиске
                {true, bidToPrice(BID1), bidToPrice(BID2), MANUAL_STRATEGY},
                {true, bidToPrice(BID1), bidToPrice(BID2), AUTO_STRATEGY}
        });
    }

    private BidsBaseRecord bidsBaseRecord;
    private DirectJooqDbSteps jooqDbSteps;
    private Long autotargetingId;

    @Before
    @Step("Создание кампании")
    public void prepare() {
        api.as(LOGIN);
        jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN);
        Long clientId = jooqDbSteps.usersSteps().getUser(LOGIN).getClientid();

        jooqDbSteps.featuresSteps().setClientFeature(clientId, FeatureNameEnum.CONTEXT_RELEVANCE_MATCH.getValue(),
                true);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        campaignId = api.userSteps.campaignSteps().addCampaign(
                new CampaignAddItemMap().defaultCampaignAddItem().withTextCampaign(new TextCampaignAddItemMap()
                        .withBiddingStrategy(initialStrategy)));

        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        if (withKeywords) {
            api.userSteps.keywordsSteps().addDefaultKeywordWithBidsAndStrategyPriority(
                    adGroupId, BID1, BID2, PriorityEnum.NORMAL);
        }
        autotargetingId = api.userSteps.keywordsSteps().addAutotargeting(adGroupId);

        bidsBaseRecord = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(autotargetingId);
        assumeThat("В bids_base выставлена нулевая ставка для бесфразного таргетинга",
                bidsBaseRecord.getPriceContext(), equalTo(bidToPrice(ZERO_PRICE)));
    }

    @Test
    public void PriceContextSettedAfterChangingStrategy() {
        api.userSteps.campaignSteps().campaignsUpdate(
                new UpdateRequestMap().withCampaigns(new CampaignUpdateItemMap().withId(campaignId)
                        .withTextCampaign(new TextCampaignUpdateItemMap().withBiddingStrategy(
                                new TextCampaignStrategyMap()
                                        .withSearch(new TextCampaignSearchStrategyMap().defaultHighestPosition())
                                        .withNetwork(new TextCampaignNetworkStrategyMap().defaultMaximumCoverage())))));

        bidsBaseRecord = jooqDbSteps.bidsBaseSteps().getBidsBaseByBidId(autotargetingId);

        assertThat("бесфразному таргетингу выставилась верная ставка на поиске: ", bidsBaseRecord.getPrice(),
                equalTo(expectedPrice));

        assertThat("бесфразному таргетингу выставилась верная ставка на сети: ", bidsBaseRecord.getPriceContext(),
                equalTo(expectedCtxPrice));
    }

    private static BigDecimal bidToPrice(Long bid) {
        return BigDecimal.valueOf(bid / 1000000.0).setScale(2);
    }
}
