package ru.yandex.autotests.directapi.changes.check;

import java.util.Arrays;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.module.jaxb.JaxbAnnotationModule;
import com.yandex.direct.api.v5.adgroups.AdGroupFieldEnum;
import com.yandex.direct.api.v5.changes.CheckFieldEnum;
import com.yandex.direct.api.v5.changes.CheckRequest;
import net.sf.cglib.beans.BeanMap;
import org.json.simple.JSONObject;
import org.json.simple.parser.JSONParser;
import org.json.simple.parser.ParseException;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.model.ModelBeanMap;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.apiclient.version5.ServiceNames;
import ru.yandex.autotests.directapi.changes.ChangesFeatures;
import ru.yandex.autotests.directapi.changes.ChangesLogins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.changes.CheckRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlApi5Error;
import ru.yandex.autotests.directapi.model.banners.BannerInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang.StringUtils.capitalize;

/**
 * Created by ginger on 04.04.15.
 * https://st.yandex-team.ru/TESTIRT-4916
 */
@Aqua.Test
@Features(ChangesFeatures.CHECK)
@Issue("https://st.yandex-team.ru/DIRECT-35862")
@Description("Проверка невалидного JSON во входных параметрах.")
public class CheckInvalidJsonTest {
    private static LogSteps log = LogSteps.getLogger(CheckInvalidJsonTest.class);

    private static String login = ChangesLogins.CHANGES_CLIENT;
    private String paramsString = "params";

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();
    private static Long campaignId;
    private static Long bannerId;
    private static Long adGroupId;
    private static String timestamp;
    private static ObjectMapper mapper;

    @Rule
    public Trashman trashman = new Trashman(api);

    @BeforeClass
    public static void createData(){
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(login);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        bannerId = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        adGroupId = api.userSteps.adGroupsSteps().adGroupsGet(new GetRequestMap()
                .withSelectionCriteria(
                        new AdGroupsSelectionCriteriaMap()
                                .withCampaignIds((long)campaignId)
                )
                .withFieldNames(
                        AdGroupFieldEnum.ID)).getAdGroups().get(0).getId();

        timestamp = api.userSteps.changesSteps().getTimestamp();
        
        mapper = new ObjectMapper();
        mapper.registerModule(new JaxbAnnotationModule());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3410")
    public void tryCallCheckWithInvalidField() throws ParseException, JsonProcessingException {
        JSONObject params = (JSONObject)(new JSONParser().parse(mapper.writeValueAsString(
                (new CheckRequestMap()
                        .withCampaignIds((long) campaignId)
                        .withTimestamp(timestamp)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS)).getBean()
        )));
        params.put(capitalize(BannerInfoMap.TYPE), "mobile");
        api.userSteps.changesSteps().shouldGetJSONErrorOnCheck(
                params, login,
                JavaOrPerlApi5Error
                        .java(new Api5Error(8000,
                                Api5ErrorDetailsJava.UNKNOWN_PARAMETER,
                                capitalize(BannerInfoMap.TYPE)))
                        .perl(new Api5Error(8000,
                                Api5ErrorDetails.FIELD_CONTAIN_UNKNOWN_FIELD,
                                paramsString,
                                capitalize(BannerInfoMap.TYPE)))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3411")
    public void tryCallCheckWithCampaignIdsNotArray() throws ParseException, JsonProcessingException {
        JSONObject params = (JSONObject)(new JSONParser().parse(mapper.writeValueAsString(
                (new CheckRequestMap()
                        .withCampaignIds((long) campaignId)
                        .withTimestamp(timestamp)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS)).getBean()
        )));
        params.put(capitalize(CheckRequestMap.CAMPAIGN_IDS), campaignId);
        api.userSteps.changesSteps().shouldGetJSONErrorOnCheck(
                params, login, new Api5Error(8000,
                Api5ErrorDetails.FIELD_MUST_CONTAIN_ARRAY,
                capitalize(CheckRequestMap.CAMPAIGN_IDS))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3412")
    public void tryCallCheckWithCampaignIdsWithStringElement() throws ParseException, JsonProcessingException {
        JSONObject params = (JSONObject)(new JSONParser().parse(mapper.writeValueAsString(
                (new CheckRequestMap()
                        .withCampaignIds((long) campaignId)
                        .withTimestamp(timestamp)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS)).getBean()
        )));
        params.put(capitalize(CheckRequestMap.CAMPAIGN_IDS),
                new Object[]{campaignId, paramsString, campaignId});
        api.userSteps.changesSteps().shouldGetJSONErrorOnCheck(
                params, login,
                JavaOrPerlApi5Error
                        .java(new Api5Error(8000,
                                Api5ErrorDetailsJava.ELEMENT_OF_ARRAY_MUST_CONTAIN_INTEGER_VALUE,
                                capitalize(CheckRequestMap.CAMPAIGN_IDS)))
                        .perl(new Api5Error(8000,
                                Api5ErrorDetails.ELEMENT_OF_ARRAY_MUST_CONTAIN_INTEGER_VALUE,
                                capitalize(CheckRequestMap.CAMPAIGN_IDS)))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3413")
    public void tryCallCheckWithStringCampaignIds() throws ParseException, JsonProcessingException {
        JSONObject params = (JSONObject)(new JSONParser().parse(mapper.writeValueAsString(
                (new CheckRequestMap()
                        .withCampaignIds((long) campaignId)
                        .withTimestamp(timestamp)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS)).getBean()
        )));
        params.put(capitalize(CheckRequestMap.CAMPAIGN_IDS), mapper.writeValueAsString(campaignId));
        api.userSteps.changesSteps().shouldGetJSONErrorOnCheck(
                params, login, new Api5Error(8000,
                Api5ErrorDetails.FIELD_MUST_CONTAIN_ARRAY,
                capitalize(CheckRequestMap.CAMPAIGN_IDS))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3414")
    public void tryCallCheckWithAdGroupIdsNotArray() throws ParseException, JsonProcessingException {
        JSONObject params = (JSONObject)(new JSONParser().parse(mapper.writeValueAsString(
                (new CheckRequestMap()
                        .withAdGroupIds(adGroupId)
                        .withTimestamp(timestamp)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS)).getBean()
        )));
        params.put(capitalize(CheckRequestMap.ADGROUP_IDS), adGroupId);
        api.userSteps.changesSteps().shouldGetJSONErrorOnCheck(
                params, login, new Api5Error(8000,
                Api5ErrorDetails.FIELD_MUST_CONTAIN_ARRAY,
                capitalize(CheckRequestMap.ADGROUP_IDS))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3415")
    public void tryCallCheckWithAdGroupIdsWithStringElement() throws ParseException, JsonProcessingException {
        JSONObject params = (JSONObject)(new JSONParser().parse(mapper.writeValueAsString(
                (new CheckRequestMap()
                        .withAdGroupIds(adGroupId)
                        .withTimestamp(timestamp)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS)).getBean()
        )));
        params.put(capitalize(CheckRequestMap.ADGROUP_IDS),
                new Object[]{adGroupId, paramsString, adGroupId});
        api.userSteps.changesSteps().shouldGetJSONErrorOnCheck(
                params, login,
                JavaOrPerlApi5Error
                        .java(new Api5Error(8000,
                                Api5ErrorDetailsJava.ELEMENT_OF_ARRAY_MUST_CONTAIN_INTEGER_VALUE,
                                capitalize(CheckRequestMap.ADGROUP_IDS)))
                        .perl(new Api5Error(8000,
                                Api5ErrorDetails.ELEMENT_OF_ARRAY_MUST_CONTAIN_INTEGER_VALUE,
                                capitalize(CheckRequestMap.ADGROUP_IDS)))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3416")
    public void tryCallCheckWithStringAdGroupIds() throws ParseException, JsonProcessingException {
        JSONObject params = (JSONObject)(new JSONParser().parse(mapper.writeValueAsString(
                (new CheckRequestMap()
                        .withAdGroupIds(adGroupId)
                        .withTimestamp(timestamp)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS)).getBean()
        )));
        params.put(capitalize(CheckRequestMap.ADGROUP_IDS), mapper.writeValueAsString(adGroupId));
        api.userSteps.changesSteps().shouldGetJSONErrorOnCheck(
                params, login, new Api5Error(8000,
                Api5ErrorDetails.FIELD_MUST_CONTAIN_ARRAY,
                capitalize(CheckRequestMap.ADGROUP_IDS))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3417")
    public void tryCallCheckWithAdIdsNotArray() throws ParseException, JsonProcessingException {
        JSONObject params = (JSONObject)(new JSONParser().parse(mapper.writeValueAsString(
                (new CheckRequestMap()
                        .withAdIds((long)bannerId)
                        .withTimestamp(timestamp)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS)).getBean()
        )));
        params.put(capitalize(CheckRequestMap.AD_IDS), bannerId);
        api.userSteps.changesSteps().shouldGetJSONErrorOnCheck(
                params, login, new Api5Error(8000,
                Api5ErrorDetails.FIELD_MUST_CONTAIN_ARRAY,
                capitalize(CheckRequestMap.AD_IDS))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3418")
    public void tryCallCheckWithAdIdsWithStringElement() throws ParseException, JsonProcessingException {
        JSONObject params = (JSONObject)(new JSONParser().parse(mapper.writeValueAsString(
                (new CheckRequestMap()
                        .withAdIds((long)bannerId)
                        .withTimestamp(timestamp)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS)).getBean()
        )));
        params.put(capitalize(CheckRequestMap.AD_IDS),
                new Object[]{bannerId, paramsString, bannerId});
        api.userSteps.changesSteps().shouldGetJSONErrorOnCheck(
                params, login,
                JavaOrPerlApi5Error
                    .java(new Api5Error(8000,
                            Api5ErrorDetailsJava.ELEMENT_OF_ARRAY_MUST_CONTAIN_INTEGER_VALUE,
                            capitalize(CheckRequestMap.AD_IDS)))
                    .perl(new Api5Error(8000,
                            Api5ErrorDetails.ELEMENT_OF_ARRAY_MUST_CONTAIN_INTEGER_VALUE,
                            capitalize(CheckRequestMap.AD_IDS)))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3419")
    public void tryCallCheckWithStringAdIds() throws ParseException, JsonProcessingException {
        JSONObject params = (JSONObject)(new JSONParser().parse(mapper.writeValueAsString(
                (new CheckRequestMap()
                        .withAdIds((long)bannerId)
                        .withTimestamp(timestamp)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS)).getBean()
        )));
        params.put(capitalize(CheckRequestMap.AD_IDS), JsonUtils.toString(bannerId));
        api.userSteps.changesSteps().shouldGetJSONErrorOnCheck(
                params, login, new Api5Error(8000,
                Api5ErrorDetails.FIELD_MUST_CONTAIN_ARRAY,
                capitalize(CheckRequestMap.AD_IDS))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3420")
    public void tryCallCheckWithFieldNamesNotArray() throws ParseException, JsonProcessingException {
        JSONObject params = (JSONObject)(new JSONParser().parse(mapper.writeValueAsString(
                (new CheckRequestMap()
                        .withAdIds((long)bannerId)
                        .withTimestamp(timestamp)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS)).getBean()
        )));
        params.put(capitalize(CheckRequestMap.FIELD_NAMES), CheckFieldEnum.CAMPAIGN_IDS);
        api.userSteps.changesSteps().shouldGetJSONErrorOnCheck(
                params, login, new Api5Error(8000,
                Api5ErrorDetails.FIELD_MUST_CONTAIN_ARRAY,
                capitalize(CheckRequestMap.FIELD_NAMES))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3421")
    public void tryCallCheckWithStringFieldNames() throws ParseException, JsonProcessingException {
        JSONObject params = (JSONObject)(new JSONParser().parse(mapper.writeValueAsString(
                (new CheckRequestMap()
                        .withAdIds((long)bannerId)
                        .withTimestamp(timestamp)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS)).getBean()
        )));
        params.put(capitalize(CheckRequestMap.FIELD_NAMES), CheckFieldEnum.CAMPAIGN_IDS);
        api.userSteps.changesSteps().shouldGetJSONErrorOnCheck(
                params, login, new Api5Error(8000,
                Api5ErrorDetails.FIELD_MUST_CONTAIN_ARRAY,
                capitalize(CheckRequestMap.FIELD_NAMES))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3422")
    public void tryCallCheckWithTimestampArray() throws ParseException, JsonProcessingException {
        JSONObject params = (JSONObject)(new JSONParser().parse(mapper.writeValueAsString(
                (new CheckRequestMap()
                        .withAdIds((long)bannerId)
                        .withTimestamp(timestamp)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS)).getBean()
        )));
        params.put(capitalize(CheckRequestMap.TIMESTAMP), new Object[]{timestamp});
        api.userSteps.changesSteps().shouldGetJSONErrorOnCheck(
                params, login, new Api5Error(8000,
                Api5ErrorDetails.FIELD_MUST_NOT_CONTAIN_ARRAY,
                capitalize(CheckRequestMap.TIMESTAMP))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3423")
    public void tryCallCheckWithInvalidTimestamp() throws ParseException, JsonProcessingException {
        JSONObject params = (JSONObject)(new JSONParser().parse(mapper.writeValueAsString(
                (new CheckRequestMap()
                        .withAdIds((long)bannerId)
                        .withTimestamp(timestamp)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS)).getBean()
        )));
        params.put(capitalize(CheckRequestMap.TIMESTAMP), 123);
        api.userSteps.changesSteps().shouldGetJSONErrorOnCheck(
                params, login,
                JavaOrPerlApi5Error
                        .java(new Api5Error(4000,
                                Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_VALUE_IN_FORMAT,
                                capitalize(CheckRequestMap.TIMESTAMP)))
                        .perl(new Api5Error(4000,
                                Api5ErrorDetails.FIELD_MUST_CONTAIN_VALUE_IN_FORMAT,
                                capitalize(CheckRequestMap.TIMESTAMP)))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3423")
    public void tryCallCheckWithInvalidTimestampNoClientLoginHeader() throws ParseException, JsonProcessingException {
        JSONObject params = (JSONObject)(new JSONParser().parse(mapper.writeValueAsString(
                (new CheckRequestMap()
                        .withAdIds((long)bannerId)
                        .withTimestamp(timestamp)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS)).getBean()
        )));

        params.put(capitalize(CheckRequestMap.TIMESTAMP), 123);
        api.as(ChangesLogins.AGENCY_1_CAMPAIGNS).userSteps.changesSteps().shouldGetJSONErrorOnCheck(
                params, null,
                new Api5Error(8000,
                        Api5ErrorDetails.NOT_CLIENT_IN_CLIENT_LOGIN,
                        capitalize(CheckRequestMap.FIELD_NAMES))
        );
    }
}
