package ru.yandex.autotests.directapi.changes.check;

import com.yandex.direct.api.v5.adgroups.AdGroupFieldEnum;
import com.yandex.direct.api.v5.changes.CheckFieldEnum;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorMessageJava;
import ru.yandex.autotests.directapi.changes.ChangesFeatures;
import ru.yandex.autotests.directapi.changes.ChangesLogins;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.changes.CheckRequestMap;
import ru.yandex.autotests.directapi.model.api5.changes.CheckResponseModifiedMap;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlApi5Error;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang.StringUtils.capitalize;

/**
 * Created by ginger on 04.04.15.
 * https://st.yandex-team.ru/TESTIRT-4916
 */
@Aqua.Test
@Features(ChangesFeatures.CHECK)
@Issue("https://st.yandex-team.ru/DIRECT-35862")
@Description("Проверка некорректных входных параметров.")
@Tag(TagDictionary.TRUNK)
public class CheckNegativeParamsTest {
    private static LogSteps log = LogSteps.getLogger(CheckNegativeParamsTest.class);

    private static final String CHECK_PARAMS = capitalize(CheckResponseModifiedMap.CAMPAIGN_IDS) + ", " +
            capitalize(CheckResponseModifiedMap.ADGROUP_IDS) + ", " +
            capitalize(CheckResponseModifiedMap.AD_IDS);

    private static String login = ChangesLogins.CHANGES_CLIENT;
    private String paramsString = "params";
    private String anotherISOFormat = "2015-05-29T13:10:46";

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();
    private static Long campaignId;
    private static Long bannerId;
    private static Long adGroupId;
    private static String timestamp;

    @Rule
    public Trashman trashman = new Trashman(api);

    @BeforeClass
    public static void createData(){
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(login);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        bannerId = api.userSteps.adsSteps().addDefaultTextAd(adGroupId);
        adGroupId = api.userSteps.adGroupsSteps().adGroupsGet(new GetRequestMap()
                .withSelectionCriteria(
                        new AdGroupsSelectionCriteriaMap()
                                .withCampaignIds((long)campaignId)
                )
                .withFieldNames(
                        AdGroupFieldEnum.ID)).getAdGroups().get(0).getId();

        timestamp = api.userSteps.changesSteps().getTimestamp();
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3441")
    public void tryJsonCallCheckWithEmptyParams(){
        api.protocol(ProtocolType.JSON).userSteps.changesSteps().shouldGetErrorOnCheck(
                new CheckRequestMap(),
            JavaOrPerlApi5Error
                .java(new Api5Error(8000,
                    Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                    capitalize(CheckRequestMap.TIMESTAMP)))
                .perl(new Api5Error(8000,
                    Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                    paramsString,
                    capitalize(CheckRequestMap.TIMESTAMP)))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3439")
    public void trySoapCallCheckWithEmptyParams(){
        api.protocol(ProtocolType.SOAP).userSteps.changesSteps().shouldGetErrorOnCheck(
                new CheckRequestMap(),
                new Api5Error(8000,
                        Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                        "CheckRequest",
                        capitalize(CheckRequestMap.TIMESTAMP))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3440")
    public void tryJsonCallCheckWithoutTimestamp(){
        api.protocol(ProtocolType.JSON).userSteps.changesSteps().shouldGetErrorOnCheck(
                new CheckRequestMap()
                        .withCampaignIds((long) campaignId)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS, CheckFieldEnum.CAMPAIGNS_STAT),
            JavaOrPerlApi5Error
                .java(new Api5Error(8000,
                    Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                    capitalize(CheckRequestMap.TIMESTAMP)))
                .perl(new Api5Error(8000,
                    Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                    paramsString,
                    capitalize(CheckRequestMap.TIMESTAMP)))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3434")
    public void trySoapCallCheckWithoutTimestamp(){
        api.protocol(ProtocolType.SOAP).userSteps.changesSteps().shouldGetErrorOnCheck(
                new CheckRequestMap()
                        .withCampaignIds((long) campaignId)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS, CheckFieldEnum.CAMPAIGNS_STAT),
                new Api5Error(8000,
                        Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                        "CheckRequest",
                        capitalize(CheckRequestMap.TIMESTAMP))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3435")
    public void tryJsonCallCheckWithoutFieldsName(){
        api.protocol(ProtocolType.JSON).userSteps.changesSteps().shouldGetErrorOnCheck(
                new CheckRequestMap()
                        .withCampaignIds((long) campaignId)
                        .withTimestamp(timestamp),
                JavaOrPerlApi5Error
                    .java(new Api5Error(8000,
                        Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        capitalize(CheckRequestMap.FIELD_NAMES)))
                    .perl(new Api5Error(8000,
                        Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                        paramsString,
                        capitalize(CheckRequestMap.FIELD_NAMES)))

        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3431")
    public void trySoapCallCheckWithoutFieldsName(){
        api.protocol(ProtocolType.SOAP).userSteps.changesSteps().shouldGetErrorOnCheck(
                new CheckRequestMap()
                        .withCampaignIds((long) campaignId)
                        .withTimestamp(timestamp),
                new Api5Error(8000,
                        Api5ErrorDetails.MISSING_REQUIRED_FIELD,
                        "CheckRequest",
                        capitalize(CheckRequestMap.FIELD_NAMES))

        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3432")
    public void tryCallCheckWithAnotherIsoFormat(){
        //another ISO Format
        //https://st.yandex-team.ru/DIRECT-35821
        api.userSteps.changesSteps().shouldGetErrorOnCheck(
                new CheckRequestMap()
                        .withCampaignIds((long) campaignId)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS, CheckFieldEnum.CAMPAIGNS_STAT)
                        .withTimestamp(anotherISOFormat),
            JavaOrPerlApi5Error
                .java(new Api5Error(4000,
                    Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_VALUE_IN_FORMAT,
                    capitalize(CheckRequestMap.TIMESTAMP)))
                .perl(new Api5Error(4000,
                    Api5ErrorDetails.FIELD_MUST_CONTAIN_VALUE_IN_FORMAT,
                    capitalize(CheckRequestMap.TIMESTAMP)))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3436")
    public void tryCallCheckWithoutIds(){
        api.userSteps.changesSteps().shouldGetErrorOnCheck(
                new CheckRequestMap()
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS, CheckFieldEnum.CAMPAIGNS_STAT)
                        .withTimestamp(timestamp),
                new Api5Error(4003,Api5ErrorDetails.ONE_OF_PARAMETERS_SHOULD_BE_SPECIFIED, CHECK_PARAMS)
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3430")
    public void tryCallCheckWithAllIds(){
        api.userSteps.changesSteps().shouldGetErrorOnCheck(
                new CheckRequestMap()
                        .withAdGroupIds(adGroupId)
                        .withAdIds((long)bannerId)
                        .withCampaignIds((long) campaignId)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS, CheckFieldEnum.CAMPAIGNS_STAT)
                        .withTimestamp(timestamp),
            JavaOrPerlApi5Error
                .java(new Api5Error(4004,
                    Api5ErrorDetailsJava.ONLY_ONE_OF_FIELDS_CAN_BE_SPECIFIED, CHECK_PARAMS)
                    .withMessageFromResource(Api5ErrorMessageJava.PUT_MORE_THAN_ONE_PARAMETER))
                .perl(new Api5Error(4004,
                    Api5ErrorDetails.ONLY_ONE_OF_FIELDS_CAN_BE_SPECIFIED, CHECK_PARAMS))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3433")
    public void tryCallCheckWithAdIDAndGroupID(){
        api.userSteps.changesSteps().shouldGetErrorOnCheck(
                new CheckRequestMap()
                        .withAdGroupIds(adGroupId)
                        .withAdIds((long) bannerId)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS, CheckFieldEnum.CAMPAIGNS_STAT)
                        .withTimestamp(timestamp),
                JavaOrPerlApi5Error
                        .java(new Api5Error(4004,
                                Api5ErrorDetailsJava.ONLY_ONE_OF_FIELDS_CAN_BE_SPECIFIED, CHECK_PARAMS)
                            .withMessageFromResource(Api5ErrorMessageJava.PUT_MORE_THAN_ONE_PARAMETER))
                        .perl(new Api5Error(4004,
                                Api5ErrorDetails.ONLY_ONE_OF_FIELDS_CAN_BE_SPECIFIED, CHECK_PARAMS))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3437")
    public void tryCallCheckWithAdIDAndCampaignID(){
        api.userSteps.changesSteps().shouldGetErrorOnCheck(
                new CheckRequestMap()
                        .withAdIds((long) bannerId)
                        .withCampaignIds((long)campaignId)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS, CheckFieldEnum.CAMPAIGNS_STAT)
                        .withTimestamp(timestamp),
            JavaOrPerlApi5Error
                .java(new Api5Error(4004,
                    Api5ErrorDetailsJava.ONLY_ONE_OF_FIELDS_CAN_BE_SPECIFIED, CHECK_PARAMS)
                    .withMessageFromResource(Api5ErrorMessageJava.PUT_MORE_THAN_ONE_PARAMETER))
                .perl(new Api5Error(4004,
                    Api5ErrorDetails.ONLY_ONE_OF_FIELDS_CAN_BE_SPECIFIED, CHECK_PARAMS))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3438")
    public void tryCallCheckWithGroupIDAndCampaignID(){
        api.userSteps.changesSteps().shouldGetErrorOnCheck(
                new CheckRequestMap()
                        .withAdGroupIds(adGroupId)
                        .withCampaignIds((long)campaignId)
                        .withFieldNames(CheckFieldEnum.CAMPAIGN_IDS, CheckFieldEnum.CAMPAIGNS_STAT)
                        .withTimestamp(timestamp),
            JavaOrPerlApi5Error
                .java(new Api5Error(4004,
                    Api5ErrorDetailsJava.ONLY_ONE_OF_FIELDS_CAN_BE_SPECIFIED, CHECK_PARAMS)
                    .withMessageFromResource(Api5ErrorMessageJava.PUT_MORE_THAN_ONE_PARAMETER))
                .perl(new Api5Error(4004,
                    Api5ErrorDetails.ONLY_ONE_OF_FIELDS_CAN_BE_SPECIFIED, CHECK_PARAMS))
        );
    }

}
