package ru.yandex.autotests.directapi.changes.checkcampaigns;

import java.sql.Timestamp;
import java.time.Instant;
import java.time.temporal.ChronoUnit;

import com.yandex.direct.api.v5.changes.CampaignChangesInEnum;
import com.yandex.direct.api.v5.changes.CampaignChangesItem;
import com.yandex.direct.api.v5.changes.CheckCampaignsResponse;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.changes.ChangesFeatures;
import ru.yandex.autotests.directapi.changes.ChangesLogins;
import ru.yandex.autotests.directapi.model.api5.changes.CampaignChangesItemMap;
import ru.yandex.autotests.directapi.model.api5.changes.CheckCampaignsRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.core.AnyOf.anyOf;
import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

@Aqua.Test
@Features(ChangesFeatures.CHECK_CAMPAIGNS)
@Issue("https://st.yandex-team.ru/DIRECT-58376")
@Description("Проверка наличия CHILDREN изменений в кампании на основе таблицы camp_aggregated_lastchange. " +
        "Может падать из-за задержек в ESS (DIRECT-107473)")
@Tag(TagDictionary.TRUNK)
public class ChangesInChildrenUseCampAggregatedTest {
    private static final String LOGIN = ChangesLogins.CHANGES_CLIENT;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    private static DirectJooqDbSteps dbSteps;

    @Rule
    public Trashman trashman = new Trashman(api);

    private Long campaignId;

    @BeforeClass
    public static void setUpClass() {
        dbSteps = api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(LOGIN);
    }

    @Before
    public void setUp() {
        // Создаем только кампанию, чтобы нам не мешал job по обновлению таблицы
        // camp_aggregated_lastchange, но при этом изменения должны возвращаться
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
    }

    @After
    public void tearDown() {
        dbSteps.campAggregatedLastChangeSteps()
                .removeLastChange(campaignId);
    }

    @Test
    public void checkCampaignsNothingChanges() {
        String timestamp = api.userSteps.changesSteps().getTimestamp();

        // Изменений по кампании быть не должно
        CheckCampaignsResponse actualCampaignsResponse = api.userSteps.changesSteps()
                .withUseCampAggregatedLastChange(true)
                .changesCheckCampaigns(
                        new CheckCampaignsRequestMap()
                                .withTimestamp(timestamp));

        CampaignChangesItem notExpectedCampaignChanges = (CampaignChangesItem) new CampaignChangesItemMap()
                .withCampaignId(campaignId)
                .withChangesIn(CampaignChangesInEnum.CHILDREN)
                .getBean();

        assertThat(
                "вернулись корректные значения",
                actualCampaignsResponse.getCampaigns(),
                not(contains(beanDifferV5(notExpectedCampaignChanges))));
    }

    @Test
    public void checkCampaignsWasOneChange() {
        String timestamp = api.userSteps.changesSteps().getTimestamp();

        // На всякий случай прибавляем 1 секунду, чтобы изменения гарантировано попали
        dbSteps.campAggregatedLastChangeSteps()
                .updateLastChange(
                        campaignId,
                        Timestamp.from(Instant.parse(timestamp).plusSeconds(1)));

        // Сейчас должны быть изменения только из-за ручной модификации таблицы camp_aggregated_lastchange,
        // хотя также возможны изменения от других компаний из-за параллельного запуска тестов
        CheckCampaignsResponse actualCampaignsResponse = api.userSteps.changesSteps()
                .withUseCampAggregatedLastChange(true)
                .changesCheckCampaigns(
                        new CheckCampaignsRequestMap()
                                .withTimestamp(timestamp));

        CampaignChangesItem expectedCampaignChanges = (CampaignChangesItem) new CampaignChangesItemMap()
                .withCampaignId(campaignId)
                .withChangesIn(CampaignChangesInEnum.CHILDREN)
                .getBean();

        assertThat(
                "вернулись корректные значения",
                actualCampaignsResponse.getCampaigns(),
                contains(beanDifferV5(expectedCampaignChanges)));
    }

    private Instant getCampAggregatedLastChangeJobEnd() {
        return dbSteps.ppcPropertiesSteps()
                .getCampAggregatedLastChangeHighBoundary(dbSteps.getCurrentPpcShard())
                .truncatedTo(ChronoUnit.SECONDS)
                .toInstant();
    }

    @Test
    public void checkCampaignsFull() {
        String timestamp = api.userSteps.changesSteps().getTimestamp();

        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        api.userSteps.adsSteps().addDefaultTextAd(adGroupId);

        // Ждем пока появится запись в таблице (Значит job-а выполнилась хотя один раз)
        ConditionFactories.CAMP_AGGREGATE_LAST_CHANGE_JOB
                .until(() -> dbSteps.campAggregatedLastChangeSteps().isRecordExists(campaignId));

        // Получаем время завершения job-ы до вызова метода
        Instant jobEndBefore = getCampAggregatedLastChangeJobEnd();

        CheckCampaignsResponse actualCampaignsResponse = api.userSteps.changesSteps()
                .withUseCampAggregatedLastChange(true)
                .changesCheckCampaigns(
                        new CheckCampaignsRequestMap()
                                .withTimestamp(timestamp));

        // Получаем время завершения job-ы после вызова метода
        Instant jobEndAfter = getCampAggregatedLastChangeJobEnd();

        CampaignChangesItem expectedCampaignChanges = (CampaignChangesItem) new CampaignChangesItemMap()
                .withCampaignId(campaignId)
                .withChangesIn(CampaignChangesInEnum.CHILDREN)
                .getBean();

        // Надеемся, что возвращаемый timestamp совпадет хотя с одни из них
        assertThat(
                "timestamp корректен",
                Instant.parse(actualCampaignsResponse.getTimestamp()),
                anyOf(
                        equalTo(jobEndBefore),
                        equalTo(jobEndAfter)));
        assertThat(
                "требуемая кампания содержится",
                actualCampaignsResponse.getCampaigns(),
                contains(beanDifferV5(expectedCampaignChanges)));
    }
}
