package ru.yandex.autotests.directapi.changes.checkdictionaries;

import java.util.ArrayList;

import com.gargoylesoftware.htmlunit.javascript.host.Map;
import org.joda.time.DateTime;
import org.json.simple.JSONObject;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.apiclient.version5.ServiceNames;
import ru.yandex.autotests.directapi.changes.ChangesFeatures;
import ru.yandex.autotests.directapi.changes.ChangesLogins;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.changes.CheckDictionariesRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlApi5Error;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang.StringUtils.capitalize;

/**
 * Created by ginger on 01.04.15.
 * https://st.yandex-team.ru/TESTIRT-4915
 */
@Aqua.Test
@Features(ChangesFeatures.CHECK_DICTIONARIES)
@Issue("https://st.yandex-team.ru/DIRECT-35862")
@Description("Проверка некорректных входных данных.")
@Tag(TagDictionary.TRUNK)
public class CheckDictionariesNegativeTest {
    private static LogSteps log = LogSteps.getLogger(CheckDictionariesNegativeTest.class);

    private static String login = ChangesLogins.CHANGES_CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();
    private static String timestamp;

    @Rule
    public Trashman trashman = new Trashman(api);
    private String paramsString = "params";
    private String invalidFieldName = "gingerField";
    private String invalidFieldValue = "gingerValue";
    private String anotherISOFormat = "2015-05-29T13:10:46";


    @BeforeClass
    public static void getTimestamp() {
        timestamp = api.userSteps.changesSteps().getTimestamp();
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3509")
    public void tryCallCheckDictionariesWithoutParams() {
        api.userSteps.changesSteps().shouldGetJSONErrorOn(ServiceNames.CHANGES, login, Action.CHECK_DICTIONARIES,
                null,
                JavaOrPerlApi5Error
                        .java(new Api5Error(8000, Api5ErrorDetailsJava.INVALID_REQUEST_FORMAT, paramsString))
                        .perl(new Api5Error(8000, Api5ErrorDetails.FIELD_CANNOT_HAVE_VALUE, paramsString, null))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3510")
    public void tryCallCheckDictionariesWithEmptyStringTimestamp() {
        JSONObject params = new JSONObject();
        params.put(capitalize(CheckDictionariesRequestMap.TIMESTAMP), "");
        api.userSteps.changesSteps().shouldGetJSONErrorOn(ServiceNames.CHANGES, login, Action.CHECK_DICTIONARIES,
                params,
                JavaOrPerlApi5Error
                        .java(new Api5Error(4000, Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_VALUE_IN_FORMAT,
                                capitalize(CheckDictionariesRequestMap.TIMESTAMP))
                        )
                        .perl(new Api5Error(4000, Api5ErrorDetails.FIELD_MUST_CONTAIN_VALUE_IN_FORMAT,
                                capitalize(CheckDictionariesRequestMap.TIMESTAMP))
                        )
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3511")
    public void tryCallCheckDictionariesWithAnotherIsoFormat() {
        //another ISO Format
        //https://st.yandex-team.ru/DIRECT-35821
        JSONObject params = new JSONObject();
        params.put(capitalize(CheckDictionariesRequestMap.TIMESTAMP), anotherISOFormat);
        api.userSteps.changesSteps().shouldGetJSONErrorOn(ServiceNames.CHANGES, login, Action.CHECK_DICTIONARIES,
                params,
                JavaOrPerlApi5Error
                        .java(new Api5Error(4000, Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_VALUE_IN_FORMAT,
                                capitalize(CheckDictionariesRequestMap.TIMESTAMP))
                        )
                        .perl(new Api5Error(4000, Api5ErrorDetails.FIELD_MUST_CONTAIN_VALUE_IN_FORMAT,
                                capitalize(CheckDictionariesRequestMap.TIMESTAMP))
                        )
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3512")
    public void tryCallCheckDictionariesWithAnotherFormatTimestamp() {
        JSONObject params = new JSONObject();
        params.put(capitalize(CheckDictionariesRequestMap.TIMESTAMP), DateTime.now().toDate().toString());
        api.userSteps.changesSteps().shouldGetJSONErrorOn(ServiceNames.CHANGES, login, Action.CHECK_DICTIONARIES,
                params,
                JavaOrPerlApi5Error
                        .java(new Api5Error(4000, Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_VALUE_IN_FORMAT,
                                capitalize(CheckDictionariesRequestMap.TIMESTAMP))
                        )
                        .perl(new Api5Error(4000, Api5ErrorDetails.FIELD_MUST_CONTAIN_VALUE_IN_FORMAT,
                                capitalize(CheckDictionariesRequestMap.TIMESTAMP))
                        )
        );
    }


    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3513")
    public void tryCheckDictionariesWithArrayTimestamp() {
        JSONObject params = new JSONObject();
        params.put(capitalize(CheckDictionariesRequestMap.TIMESTAMP), new Object[]{timestamp});
        api.userSteps.changesSteps().shouldGetJSONErrorOnCheckCampaigns(
                params, login, new Api5Error(8000,
                        Api5ErrorDetails.FIELD_MUST_NOT_CONTAIN_ARRAY,
                        capitalize(CheckDictionariesRequestMap.TIMESTAMP))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3514")
    public void tryCallCheckDictionariesWithIntegerTimestamp() {
        JSONObject params = new JSONObject();
        params.put(capitalize(CheckDictionariesRequestMap.TIMESTAMP), 666);
        api.userSteps.changesSteps().shouldGetJSONErrorOn(ServiceNames.CHANGES, login, Action.CHECK_DICTIONARIES,
                params,
                JavaOrPerlApi5Error
                        .java(new Api5Error(4000, Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_VALUE_IN_FORMAT,
                                capitalize(CheckDictionariesRequestMap.TIMESTAMP))
                        )
                        .perl(new Api5Error(4000, Api5ErrorDetails.FIELD_MUST_CONTAIN_VALUE_IN_FORMAT,
                                capitalize(CheckDictionariesRequestMap.TIMESTAMP))
                        )
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3515")
    public void tryCallCheckDictionariesWithUnknownField() {
        JSONObject params = new JSONObject();
        params.put(invalidFieldName, invalidFieldValue);
        params.put(capitalize(CheckDictionariesRequestMap.TIMESTAMP), timestamp);
        api.userSteps.changesSteps().shouldGetJSONErrorOn(ServiceNames.CHANGES, login, Action.CHECK_DICTIONARIES,
                params,
                JavaOrPerlApi5Error
                        .java(new Api5Error(8000, Api5ErrorDetailsJava.UNKNOWN_PARAMETER, invalidFieldName))
                        .perl(new Api5Error(8000, Api5ErrorDetails.FIELD_CONTAIN_UNKNOWN_FIELD, paramsString,
                                invalidFieldName)
                        )
        );
    }
}
