package ru.yandex.autotests.directapi.changes.checkdictionaries;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.yandex.direct.api.v5.changes.CheckDictionariesResponse;
import com.yandex.direct.api.v5.general.YesNoEnum;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.db.steps.PpcPropertiesSteps;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.changes.ChangesFeatures;
import ru.yandex.autotests.directapi.changes.ChangesLogins;
import ru.yandex.autotests.directapi.model.api5.changes.CheckDictionariesRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.beandiffer.matchvariation.DefaultMatchVariation;
import ru.yandex.autotests.irt.testutils.beandiffer.matchvariation.MatchVariation;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanEquivalent;

/**
 * Created by ginger on 01.04.15.
 * https://st.yandex-team.ru/TESTIRT-4915
 */
@Aqua.Test
@Features(ChangesFeatures.CHECK_DICTIONARIES)
@Issue("https://st.yandex-team.ru/DIRECT-35862")
@Description("Проверка корректных выходных данных.")
@Tag(TagDictionary.TRUNK)
public class CheckDictionariesPositiveTest {
    private static String timestampAfterAllChangings;
    private static String timestampBeforeInterestsChanging;
    static String dateTimePattern = "yyyy-MM-dd\'T\'HH:mm:ss\'Z\'";
    static String regexpDateTimePattern = "\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}Z";

    static DateTimeFormatter methodFormatter;

    static {
        methodFormatter = DateTimeFormat.forPattern(dateTimePattern);
    }

    private static final String LOGIN = ChangesLogins.CHANGES_CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    @BeforeClass
    public static void getTimestamps(){
        DirectJooqDbSteps dbSteps = api.userSteps.getDirectJooqDbSteps().useShardForLogin(LOGIN);

        Long timestampTimeBeforeInterestsChanging = dbSteps.interestSteps().getTargetingCategoriesRecords().stream()
                .map(interest -> interest.getLastChange().getTime())
                .max(Long::compareTo).get();
        timestampBeforeInterestsChanging =
                new DateTime(timestampTimeBeforeInterestsChanging).minusHours(3).toString(methodFormatter);

        timestampAfterAllChangings = DateTime.now().toString(methodFormatter);
    }
    /*  Время из БД можно получить вызовом
        m ts:ppcdict 'select * from ppc_properties where name in("geo_regions_update_time", "geo_timezones_update_time")'
        от полученного времени нужно отнять три часа.
    */
    private String timestampBeforeRegionsChanging = "2016-10-13T14:00:34Z";
    private String timestampBeforeTimeZonesChanging = "2016-10-13T14:00:55Z";

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3516")
    public void callCheckDictionariesWithoutTimestamp(){
        CheckDictionariesResponse actualCheckDictionariesResponse =
                api.userSteps.changesSteps().changesCheckDictionaries(
                        new CheckDictionariesRequestMap()
                );
        MatchVariation variation = new DefaultMatchVariation()
                .forFields(CheckDictionariesRequestMap.TIMESTAMP).useMatcher(notNullValue());
        CheckDictionariesResponse expectedCheckDictionariesResponse = new CheckDictionariesResponse();
        assertThat("вернулись корректные поля", actualCheckDictionariesResponse,
                beanEquivalent(expectedCheckDictionariesResponse).withVariation(variation));
        checkTimesmamp(actualCheckDictionariesResponse.getTimestamp());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3517")
    public void callCheckDictionariesWithTimestampBeforeRegionsChanging(){
        CheckDictionariesResponse actualCheckDictionariesResponse =
                api.userSteps.changesSteps().changesCheckDictionaries(
                        new CheckDictionariesRequestMap()
                                .withTimestamp(timestampBeforeRegionsChanging)
                );
        MatchVariation variation = new DefaultMatchVariation()
                .forFields(CheckDictionariesRequestMap.TIMESTAMP).useMatcher(notNullValue());
        CheckDictionariesResponse expectedCheckDictionariesResponse = new CheckDictionariesResponse();
        expectedCheckDictionariesResponse.setRegionsChanged(YesNoEnum.YES);
        assertThat("вернулись корректные поля", actualCheckDictionariesResponse,
                beanEquivalent(expectedCheckDictionariesResponse).withVariation(variation));
        checkTimesmamp(actualCheckDictionariesResponse.getTimestamp());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("3518")
    public void callCheckDictionariesWithTimestampBeforeTimeZonesChanging(){
        CheckDictionariesResponse actualCheckDictionariesResponse =
                api.userSteps.changesSteps().changesCheckDictionaries(
                        new CheckDictionariesRequestMap()
                                .withTimestamp(timestampBeforeTimeZonesChanging)
                );
        MatchVariation variation = new DefaultMatchVariation()
                .forFields(CheckDictionariesRequestMap.TIMESTAMP).useMatcher(notNullValue());
        CheckDictionariesResponse expectedCheckDictionariesResponse = new CheckDictionariesResponse();
        expectedCheckDictionariesResponse.setTimeZonesChanged(YesNoEnum.YES);
        assertThat("вернулись корректные поля", actualCheckDictionariesResponse,
                beanEquivalent(expectedCheckDictionariesResponse).withVariation(variation));
        checkTimesmamp(actualCheckDictionariesResponse.getTimestamp());
    }

    @Test
    @Description("https://st.yandex-team.ru/DIRECT-61013")
    public void callCheckDictionariesWithTimestampBeforeInterestsChanging(){
        CheckDictionariesResponse actualCheckDictionariesResponse =
            api.userSteps.changesSteps().changesCheckDictionaries(
                    new CheckDictionariesRequestMap()
                            .withTimestamp(timestampBeforeInterestsChanging)
            );
        MatchVariation variation = new DefaultMatchVariation()
                .forFields(CheckDictionariesRequestMap.TIMESTAMP).useMatcher(notNullValue());
        CheckDictionariesResponse expectedCheckDictionariesResponse = new CheckDictionariesResponse();
        expectedCheckDictionariesResponse.setInterestsChanged(YesNoEnum.YES);
        assertThat("вернулись корректные поля", actualCheckDictionariesResponse,
                beanEquivalent(expectedCheckDictionariesResponse).withVariation(variation));
        checkTimesmamp(actualCheckDictionariesResponse.getTimestamp());
    }
    @Test
    @Description("https://st.yandex-team.ru/DIRECT-61013")
    public void callCheckDictionariesWithTimestampBeforeAllChanging(){
        CheckDictionariesResponse actualCheckDictionariesResponse =
                api.userSteps.changesSteps().changesCheckDictionaries(
                        new CheckDictionariesRequestMap()
                                .withTimestamp(timestampAfterAllChangings)
                );
        MatchVariation variation = new DefaultMatchVariation()
                .forFields(CheckDictionariesRequestMap.TIMESTAMP).useMatcher(notNullValue());
        CheckDictionariesResponse expectedCheckDictionariesResponse = new CheckDictionariesResponse();
        expectedCheckDictionariesResponse.setRegionsChanged(YesNoEnum.NO);
        expectedCheckDictionariesResponse.setTimeZonesChanged(YesNoEnum.NO);
        expectedCheckDictionariesResponse.setInterestsChanged(YesNoEnum.NO);
        assertThat("вернулись корректные поля", actualCheckDictionariesResponse,
                beanEquivalent(expectedCheckDictionariesResponse).withVariation(variation));
        checkTimesmamp(actualCheckDictionariesResponse.getTimestamp());
    }

    void checkTimesmamp(String timestamp) {
        Assert.assertTrue("таймстэмп не соответствует формату " + dateTimePattern,
                timestamp.matches(regexpDateTimePattern));
    }
}
