package ru.yandex.autotests.directapi.clients.get;

import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.clients.ClientFieldEnum;
import com.yandex.direct.api.v5.clients.GetResponse;
import com.yandex.direct.api.v5.general.LangEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.generalclients.EmailSubscriptionEnum;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.UsersRecord;
import ru.yandex.autotests.directapi.clients.ClientsFeatures;
import ru.yandex.autotests.directapi.clients.ClientsLogins;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.darkside.connection.Semaphore;
import ru.yandex.autotests.directapi.model.api5.clients.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.ClientGetItemMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.EmailSubscriptionItemMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.NotificationGetMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Collections.singletonList;
import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by buhter on 10/02/16.
 * https://st.yandex-team.ru/TESTIRT-8440
 */
@Aqua.Test
@Features(ClientsFeatures.GET)
@Issue("https://st.yandex-team.ru/DIRECT-49746")
@Description("Проверки получения поля Notification")
@RunWith(Parameterized.class)
public class GetNotificationTest {
    @ClassRule
    public static ApiSteps api = new ApiSteps().as(Logins.LOGIN_SUPER);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter
    public String login;

    @Parameterized.Parameter(1)
    public NotificationGetMap expected;

    @Parameterized.Parameters(name = "Логин - {0}")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {Logins.LOGIN_MAIN, createNotificationGetMap(Logins.LOGIN_MAIN, "", true)},
                {ClientsLogins.LOGIN_PHONE, createNotificationGetMap(ClientsLogins.LOGIN_PHONE, "+7000*****37", true)},
                {ClientsLogins.LOGIN_AGENCY, createNotificationGetMap(ClientsLogins.LOGIN_AGENCY, "", false)},
        });
    }

    @Before
    @Step("Подготовка тестовых данных")
    public void clear() {
        api.as(login);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(login);
    }

    @Test
    public void getNotificationTest() {
        GetResponse getResponse = api.userSteps.clientsStepsV5().clientsGet(
                new GetRequestMap().withFieldNames(ClientFieldEnum.NOTIFICATION));
        assertThat("полученные поля совпадают с ожидаемыми", getResponse.getClients(),
                beanDifferV5(singletonList(new ClientGetItemMap().withNotification(expected).getBean())));
    }

    private static NotificationGetMap createNotificationGetMap(String login, String phone,
            boolean withEmailSubscriptions)
    {
        UsersRecord usersRecord = api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(login)
                .usersSteps()
                .getUser(login);
        NotificationGetMap notificationGetMap = new NotificationGetMap()
                .withLang(LangEnum.fromValue(usersRecord.getLang().toUpperCase()))
                .withEmail(usersRecord.getEmail())
                .withSmsPhoneNumber(phone);
        if (withEmailSubscriptions) {
            notificationGetMap.withEmailSubscriptions(
                    new EmailSubscriptionItemMap(EmailSubscriptionEnum.RECEIVE_RECOMMENDATIONS,
                            YesNoEnum.fromValue(usersRecord.getSendnews().getLiteral().toUpperCase())),
                    new EmailSubscriptionItemMap(EmailSubscriptionEnum.TRACK_MANAGED_CAMPAIGNS,
                            YesNoEnum.fromValue(usersRecord.getSendaccnews().getLiteral().toUpperCase())),
                    new EmailSubscriptionItemMap(EmailSubscriptionEnum.TRACK_POSITION_CHANGES,
                            YesNoEnum.fromValue(usersRecord.getSendwarn().getLiteral().toUpperCase()))
            );
        }
        return notificationGetMap;
    }
}
