package ru.yandex.autotests.directapi.clients.get;

import java.util.List;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.campaigns.CampaignFieldEnum;
import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import com.yandex.direct.api.v5.clients.ClientFieldEnum;
import com.yandex.direct.api.v5.clients.GetResponse;
import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.generalclients.ClientSettingGetEnum;
import com.yandex.direct.api.v5.generalclients.ClientSettingGetItem;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampaignsRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.clients.ClientsFeatures;
import ru.yandex.autotests.directapi.common.api45mng.CreateNewSubclientResponse;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.clients.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by onotole on 9/5/16.
 * https://st.yandex-team.ru/TESTIRT-10114
 */
@Aqua.Test
@Features(ClientsFeatures.GET)
@Issue("https://st.yandex-team.ru/DIRECT-54913")
@Description("Проверки выдачи параметра SharedAccountEnabled для разных случаев")
public class GetSharedAccountEnabledTest {
    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    private static final Class managerWsdl = ru.yandex.autotests.directapi.common.api45mng.APIPort_PortType.class;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().wsdl(managerWsdl);

    @Rule
    public Trashman trashman = new Trashman(api);

    private boolean isSharedAccountEnabled(String client) {

        YesNoEnum realValue = null;
        GetResponse getResponse = api.as(Logins.SUPER_LOGIN, client).userSteps.clientsStepsV5().clientsGet(
                new GetRequestMap().withFieldNames(ClientFieldEnum.SETTINGS));

        List<ClientSettingGetItem> settingsList = getResponse.getClients().get(0).getSettings();
        for (ClientSettingGetItem settingItemGet : settingsList) {
            if (settingItemGet.getOption().equals(ClientSettingGetEnum.SHARED_ACCOUNT_ENABLED)) {
                realValue = settingItemGet.getValue();
            }
        }
        return YesNoEnum.YES.equals(realValue);
    }

    private boolean isSharedAccountEnabled(String client, String agencyLogin) {
        YesNoEnum realValue = null;
        GetResponse getResponse = api.as(agencyLogin).userSteps.clientsStepsV5().clientsGet(
                new GetRequestMap().withFieldNames(ClientFieldEnum.SETTINGS), client);

        List<ClientSettingGetItem> settingsList = getResponse.getClients().get(0).getSettings();
        for (ClientSettingGetItem settingItemGet : settingsList) {
            if (settingItemGet.getOption().equals(ClientSettingGetEnum.SHARED_ACCOUNT_ENABLED)) {
                realValue = settingItemGet.getValue();
            }
        }
        return YesNoEnum.YES.equals(realValue);
    }

    private void disableSharedAccountForAllCampaigns() {
        List<CampaignGetItem> campaignsList = api.userSteps.campaignSteps().campaignsGet(
                new ru.yandex.autotests.directapi.model.api5.campaigns.GetRequestMap().withSelectionCriteria(
                        new CampaignsSelectionCriteriaMap()).withFieldNames(CampaignFieldEnum.ID)).getCampaigns();
        List<Long> campaignsIDList = campaignsList.stream().map(CampaignGetItem::getId).collect(Collectors.toList());
        if (campaignsIDList.isEmpty()) {
            return;
        }

        for (Long campaignID : campaignsIDList) {
            disableSharedAccount(campaignID);
        }
    }

    private void disableSharedAccount(Long campaignId) {
        api.userSteps.campaignFakeSteps().resetWalletCreateDate(campaignId);

        Integer shard = api.userSteps.getDirectJooqDbSteps().shardingSteps().getShardByCid(campaignId);
        DirectJooqDbSteps jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShard(shard);
        CampaignsRecord campaign = jooqDbSteps.campaignsSteps().getCampaignById(campaignId);
        campaign.setWalletCid(0L);
        jooqDbSteps.campaignsSteps().updateCampaigns(campaign);
    }

    @Test
    public void onlyClientsSharedAccount() {
        String client = Logins.CLIENT_SHARED_ACCOUNT;
        api.as(client);
        disableSharedAccountForAllCampaigns();
        api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        assumeThat("Общий счет отключен", isSharedAccountEnabled(client), equalTo(false));
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().enableSharedAccount(client);
        assertThat("Общий счет подключился", isSharedAccountEnabled(client), equalTo(true));
    }

    @Test
    public void oneAgencySharedAccount() {
        String agencyLogin = Logins.AGENCY_EUR;
        api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().enableToCreateSubClients(agencyLogin);
        CreateNewSubclientResponse createNewSubclientResponse = api.as(agencyLogin).userSteps.clientSteps()
                .createNewAgencySubClient("at-testcl-sa", agencyLogin, Currency.EUR);
        String client = createNewSubclientResponse.getLogin();
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        api.as(Logins.SUPER_LOGIN, client);
        assumeThat("общий счет подключен", isSharedAccountEnabled(client), equalTo(true));

        api.as(Logins.SUPER_LOGIN, client);
        disableSharedAccount(campaignID);
        assertThat("Общий счет отключился", isSharedAccountEnabled(client), equalTo(false));

        api.as(agencyLogin).userSteps.financeSteps().enableSharedAccount(client);
        assertThat("Общий счет подключился", isSharedAccountEnabled(client), equalTo(true));

    }

    @Test
    public void agentHasPermissionsToClientSharedAccount() {
        String agencyLogin = Logins.AGENCY_ANOTHER;
        api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().enableToCreateSubClients(agencyLogin);
        CreateNewSubclientResponse createNewSubclientResponse = api.as(agencyLogin).userSteps.clientSteps()
                .createNewAgencySubClient("at-test-cl-sa", agencyLogin, Currency.RUB);
        String client = createNewSubclientResponse.getLogin();
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        api.userSteps.financeSteps().enableSharedAccount(client);
        assumeThat("Агенство имеет права проверить общий счет у сабклиента",
                isSharedAccountEnabled(client, agencyLogin), equalTo(true));
        api.as(Logins.SUPER_LOGIN, client);
        disableSharedAccount(campaignId);
        assertThat("Агенство не имеет права проверить общий счет у сабклиента",
                isSharedAccountEnabled(client, agencyLogin), equalTo(false));
    }


}
