package ru.yandex.autotests.directapi.clients.get;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.clients.ClientFieldEnum;
import com.yandex.direct.api.v5.clients.GetResponse;
import com.yandex.direct.api.v5.general.CurrencyEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.generalclients.ClientGetItem;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.AccountScoreRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.ClientsRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.UsersRecord;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.clients.ClientsFeatures;
import ru.yandex.autotests.directapi.clients.ClientsLogins;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.clients.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.generalclients.ClientGetItemMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by buhter on 10/02/16.
 * https://st.yandex-team.ru/TESTIRT-8440
 */
@Aqua.Test
@Features(ClientsFeatures.GET)
@Issue("https://st.yandex-team.ru/DIRECT-49746")
@Description("Проверки получения 'простых' полей")
@RunWith(Parameterized.class)
public class GetSimpleFieldsTest {
    private static final String PATTERN = "yyyy-MM-dd";
    @ClassRule
    public static ApiSteps api = new ApiSteps().as(Logins.LOGIN_SUPER);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter
    public String login;

    private ClientGetItemMap expected;
    private BigDecimal expectedAccountScore;
    private ClientFieldEnum[] testFields;

    @Parameterized.Parameters(name = "Логин - {0}")
    public static Collection<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {Logins.LOGIN_KZT},
                {Logins.LOGIN_EUR},
                {Logins.LOGIN_RUB},
                {ClientsLogins.CLIENT_BYN}
        });
    }

    @Before
    @Step("Подготовка тестовых данных")
    public void clear() {
        api.as(login);
        Long clientId = Long.valueOf(User.get(login).getClientID());
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(login);

        UsersRecord usersRecord = api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(login)
                .usersSteps()
                .getUser(login);
        ClientsRecord clientsRecord = api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(login)
                .clientsSteps()
                .getClients(clientId);

        AccountScoreRecord accountScoreRecord = api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(login)
                .accountScoreSteps()
                .getLastAccountScore(clientId);
        expectedAccountScore = accountScoreRecord == null ? null : accountScoreRecord.getScore().stripTrailingZeros();

        String expectedCreatedAt
                = DateTimeFormat.forPattern(PATTERN).print(new DateTime(usersRecord.getCreatetime().longValue() *
                1000));

        CurrencyEnum expectedCurrency = clientsRecord.getWorkCurrency() == null
                ? CurrencyEnum.YND_FIXED : CurrencyEnum.fromValue(clientsRecord.getWorkCurrency().getLiteral());

        Long countryId = clientsRecord.getCountryRegionId();

        testFields = new ClientFieldEnum[]{
                ClientFieldEnum.ARCHIVED
                , ClientFieldEnum.LOGIN
                , ClientFieldEnum.CLIENT_ID
                , ClientFieldEnum.CREATED_AT
                , ClientFieldEnum.CURRENCY
                , ClientFieldEnum.PHONE
                , ClientFieldEnum.CLIENT_INFO
                , ClientFieldEnum.ACCOUNT_QUALITY
                , ClientFieldEnum.COUNTRY_ID
        };
        String phone = usersRecord.getPhone();
        expected = new ClientGetItemMap()
                .withArchived(YesNoEnum.fromValue(usersRecord.getStatusarch().getLiteral().toUpperCase()))
                .withLogin(login)
                .withClientId(clientId)
                .withCreatedAt(expectedCreatedAt)
                .withCurrency(expectedCurrency)
                .withPhone(phone == null ? "" : phone)
                .withAccountQuality(expectedAccountScore)
                .withClientInfo(usersRecord.getFio())
                .withCountryId(countryId.intValue());
    }

    @Test
    public void simpleFieldsTest() {
        GetResponse getResponse = api.userSteps.clientsStepsV5().clientsGet(
                new GetRequestMap().withFieldNames(testFields));

        assertThat("полученные поля совпадают с ожидаемыми", getResponse.getClients()
                , beanDifferV5(
                        Arrays.asList((ClientGetItem) expected.getBean())
                ));
    }
}
