package ru.yandex.autotests.directapi.dictionaries.get.positive.currencies;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.util.streamex.StreamEx;

import com.google.common.collect.Lists;
import com.yandex.direct.api.v5.dictionaries.ConstantsItem;
import com.yandex.direct.api.v5.dictionaries.CurrenciesItem;
import com.yandex.direct.api.v5.dictionaries.DictionaryNameEnum;
import com.yandex.direct.api.v5.dictionaries.GetResponse;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.dictionaries.DictionariesFeatures;
import ru.yandex.autotests.directapi.dictionaries.DictionariesLogins;
import ru.yandex.autotests.directapi.model.api5.dictionaries.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Title;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.everyItem;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.isEmptyOrNullString;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.hasItem;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by proxeter on 26.01.16.
 *
 * https://st.yandex-team.ru/TESTIRT-8316
 */
@Aqua.Test
@Title("Получение словарей с константами")
@Features(DictionariesFeatures.GET)
@Issue("https://st.yandex-team.ru/DIRECT-41252")
public class GetCurrenciesDictionaryTest {

    private static final String LOGIN = DictionariesLogins.DICTIONARIES_LOGIN;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @Rule
    public Trashman trashman = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private List<CurrenciesItem> currencies;

    @Before
    public void before() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        GetResponse dictionaries = api.userSteps.dictionariesSteps().getDictionaries(
                new GetRequestMap().withDictionaryNames(DictionaryNameEnum.CURRENCIES)
        );

        currencies = dictionaries.getCurrencies();

        assumeThat("получен словарь с валютами", currencies, hasSize(greaterThan(0)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("9609")
    public void getCurrency() {
        List<String> currencies =
                this.currencies.stream().map(CurrenciesItem::getCurrency).collect(Collectors.toList());

        assertThat("валюта присутствуют", currencies, everyItem(not(isEmptyOrNullString())));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("9610")
    public void getPropertyName() {
        List<List<ConstantsItem>> propertiesList =
                currencies.stream().map(CurrenciesItem::getProperties).collect(Collectors.toList());

        List<ConstantsItem> allProperties = propertiesList.stream().flatMap(List::stream).collect(Collectors.toList());
        List<String> names = allProperties.stream().map(ConstantsItem::getName).collect(Collectors.toList());

        assertThat("свойство 'имя' присутствует", names, everyItem(not(isEmptyOrNullString())));
        assertThat("найдена добавленная константа", names, hasItem("MinCpcCpaPerformance"));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("9611")
    public void getPropertyValue() {
        List<List<ConstantsItem>> propertiesList =
                currencies.stream().map(CurrenciesItem::getProperties).collect(Collectors.toList());

        List<ConstantsItem> allProperties = Lists.newArrayList();
        propertiesList.stream().forEach(allProperties::addAll);

        List<String> values = allProperties.stream().map(ConstantsItem::getValue).collect(Collectors.toList());

        assertThat("свойство 'значение' присутствует", values, everyItem(not(isEmptyOrNullString())));
    }

    @Test
    public void noYndFixedCurrencyTest() {
        Optional<String> yndFixedItem = StreamEx.of(currencies)
                .map(CurrenciesItem::getCurrency)
                .findAny("YND_FIXED"::equals);

        assertThat("Валюта YND_FIXED не должна возвращаться", yndFixedItem.isPresent(), is(false));
    }

    @Test
    public void currenciesDoNotContainRateProperty() {
        Optional<String> rateProperty = StreamEx.of(currencies)
                .flatCollection(CurrenciesItem::getProperties)
                .map(ConstantsItem::getName)
                .findAny("Rate"::equals);

        assertThat("Валюты не должны содержать поле Rate", rateProperty.isPresent(), is(false));
    }

    @Test
    public void currenciesDoNotContainRateWithVATProperty() {
        Optional<String> rateWithVATProperty = StreamEx.of(currencies)
                .flatCollection(CurrenciesItem::getProperties)
                .map(ConstantsItem::getName)
                .findAny("RateWithVAT"::equals);

        assertThat("Валюты не должны содержать поле RateWithVAT", rateWithVATProperty.isPresent(), is(false));
    }
}
