package ru.yandex.autotests.directapi.dynamictextadtargets.add;

import java.util.List;

import com.yandex.direct.api.v5.dynamictextadtargets.StringConditionOperatorEnum;
import com.yandex.direct.api.v5.dynamictextadtargets.WebpageConditionOperandEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsFeatures;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsLogins;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.DeleteRequestMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageAddItemMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageConditionMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by buhter on 02/09/15.
 * https://st.yandex-team.ru/TESTIRT-6666
 */
@Aqua.Test
@Features(DynamicTextAdTargetsFeatures.ADD)
@Issue("https://st.yandex-team.ru/DIRECT-43826")
public class AddTest {
    private static String client = DynamicTextAdTargetsLogins.SINGLE_CLIENT_ADD;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(client);

    private static Long pid;
    private static Long cid;
    private static int shard;

    @Rule
    public Trashman trashman = new Trashman(api);


    @BeforeClass
    public static void initTest() {
        cid = api.userSteps.campaignSteps().addDefaultDynamicTextCampaign();
        shard = api.userSteps.clientFakeSteps().getUserShard(client);
    }

    @Before
    public void createGroup() {
        pid = api.userSteps.adGroupsSteps().addDefaultGroupDynamic(cid);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
    }

    @Test
    public void addWithValidParams() {
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withWebpages(new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name")
                        .withBid(30000000l)
                ),
                ExpectedResult.success()
        );
    }


    @Test
    @Description("Вызвать метод добавления условий нацеливания с одинаковыми Webpages")
    public void tryAddSameWebpagesWithSameConditionByTwoRequests() {
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withWebpages(
                        new WebpageAddItemMap()
                                .withConditions(new WebpageConditionMap()
                                        .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                        .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                        .withArguments("ginger")
                                )
                                .withName("Name1")
                                .withBid(30000000l)
                                .withAdGroupId(pid)
                ),
                ExpectedResult.success()
        );
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withWebpages(
                        new WebpageAddItemMap()
                                .withConditions(new WebpageConditionMap()
                                        .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                        .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                        .withArguments("ginger")
                                )
                                .withName("Name1")
                                .withBid(30000000l)
                                .withAdGroupId(pid)),
                ExpectedResult.errors(new Notification(6000,
                        Api5ErrorDetails.WEBPAGES_WITH_THIS_CONDITIONS_ALREADY_EXIST_IN_GROUP))
        );
    }

    @Test
    public void addWithSameNameAndAnotherCondition() {
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withWebpages(new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name")
                        .withConditions(
                                new WebpageConditionMap()
                                        .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                        .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                        .withArguments("ginger"))
                        .withBid(30000000l)
                ),
                ExpectedResult.success()
        );
        List<Long> webpagesIds = api.userSteps.getDirectJooqDbSteps().useShard(shard).bidsDynamicSteps()
                .getBidsDynamicCondIdsByPid(pid);
        assumeThat("в группе одно условие нацеливания", webpagesIds.size(), equalTo(1));
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withWebpages(new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name")
                        .withConditions(
                                new WebpageConditionMap()
                                        .withOperand(WebpageConditionOperandEnum.PAGE_CONTENT)
                                        .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                        .withArguments("ginger"))
                        .withBid(30000000l)
                ),
                ExpectedResult.success()
        );
        webpagesIds = api.userSteps.getDirectJooqDbSteps().useShard(shard).bidsDynamicSteps()
                .getBidsDynamicCondIdsByPid(pid);
        assertThat("условие не перезатёрлось, а добавилось новое", webpagesIds.size(), equalTo(2));
    }


    @Test
    public void addWithSameConditionAndAnotherName() {
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withWebpages(new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name1")
                        .withConditions(
                                new WebpageConditionMap()
                                        .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                        .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                        .withArguments("ginger"))
                        .withBid(30000000l)
                ),
                ExpectedResult.success()
        );
        List<Long> webpagesIds = api.userSteps.getDirectJooqDbSteps().useShard(shard).bidsDynamicSteps()
                .getBidsDynamicCondIdsByPid(pid);
        assumeThat("в группе одно условие нацеливания", webpagesIds.size(), equalTo(1));
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withWebpages(new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name2")
                        .withConditions(
                                new WebpageConditionMap()
                                        .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                        .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                        .withArguments("ginger"))
                        .withBid(30000000l)
                ),
                ExpectedResult.errors(new Notification(6000,
                        Api5ErrorDetails.WEBPAGES_WITH_THIS_CONDITIONS_ALREADY_EXIST_IN_GROUP))
        );
    }


    @Test
    public void tryAddPagesAllConditionAndPagesSubsetByTwoRequestsFirstAll() {
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withWebpages(
                        new WebpageAddItemMap()
                                .withAdGroupId(pid)
                                .withName("Name")
                                .withBid(30000000l)
                ),
                ExpectedResult.success()
        );
        JavaOrPerlExpectedResult expectedResult = JavaOrPerlExpectedResult
                .java(ExpectedResult.errors(new Notification(6000,
                        Api5ErrorDetails.WEBPAGE_WITH_ALL_CONDITION_CAN_NOT_BE_WITH_ANOTHER_CONDITION)
                        .withDetails("Условие нацеливания с правилом `Все страницы` не может комбинироваться с " +
                                "другими условиями в группе")))
                .perl(ExpectedResult.errors(new Notification(6000,
                        Api5ErrorDetails.WEBPAGE_WITH_ALL_CONDITION_CAN_NOT_BE_WITH_ANOTHER_CONDITION)));
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withWebpages(
                        new WebpageAddItemMap()
                                .withAdGroupId(pid)
                                .withName("Name")
                                .withConditions(
                                        new WebpageConditionMap()
                                                .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                                .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                                .withArguments("ginger"))
                                .withBid(30000000l)
                ),
                expectedResult
        );
    }

    @Test
    public void tryAddPagesAllConditionAndPagesSubsetByTwoRequestsFirstSubset() {
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withWebpages(
                        new WebpageAddItemMap()
                                .withAdGroupId(pid)
                                .withName("Name")
                                .withConditions(
                                        new WebpageConditionMap()
                                                .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                                .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                                .withArguments("ginger"))
                                .withBid(30000000l)
                ),
                ExpectedResult.success()
        );
        JavaOrPerlExpectedResult expectedResult = JavaOrPerlExpectedResult
                .java(ExpectedResult.errors(new Notification(6000,
                        Api5ErrorDetails.WEBPAGE_WITH_ALL_CONDITION_CAN_NOT_BE_WITH_ANOTHER_CONDITION)
                        .withDetails("Условие нацеливания с правилом `Все страницы` не может комбинироваться с " +
                                "другими условиями в группе"))
                )
                .perl(ExpectedResult.errors(new Notification(6000,
                        Api5ErrorDetails.WEBPAGE_WITH_ALL_CONDITION_CAN_NOT_BE_WITH_ANOTHER_CONDITION))
                );

        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withWebpages(
                        new WebpageAddItemMap()
                                .withAdGroupId(pid)
                                .withName("Name")
                                .withBid(30000000l)
                ),
                expectedResult);
    }

    @Test
    public void tryAddSecondPagesAllCondition() {
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withWebpages(
                        new WebpageAddItemMap()
                                .withAdGroupId(pid)
                                .withName("Name")
                                .withBid(30000000l)
                ),
                ExpectedResult.success()
        );
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withWebpages(
                        new WebpageAddItemMap()
                                .withAdGroupId(pid)
                                .withName("Name")
                                .withBid(30000000l)
                ),
                ExpectedResult.errors(new Notification(6000,
                        Api5ErrorDetails.WEBPAGES_WITH_THIS_CONDITIONS_ALREADY_EXIST_IN_GROUP))
        );
    }

    @Test
    public void addSameWebpageAfterDeleting() {
        Long idBeforeDeleting = api.userSteps.dynamicTextAdTargetsSteps().addWebpage(
                new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name")
                        .withBid(30000000l)
        );
        api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsDelete(
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap().withIds(idBeforeDeleting))
        );
        Long idAfterDeleting = api.userSteps.dynamicTextAdTargetsSteps().addWebpage(
                new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name")
                        .withBid(30000000l)
        );
        assertThat("ID условия совпадает с удалённым", idAfterDeleting, equalTo(idBeforeDeleting));
    }

}
