package ru.yandex.autotests.directapi.dynamictextadtargets.add;

import com.yandex.direct.api.v5.dynamictextadtargets.StringConditionOperatorEnum;
import com.yandex.direct.api.v5.dynamictextadtargets.WebpageConditionOperandEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsFeatures;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsLogins;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageAddItemMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageConditionMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qa.balance.tools.common.utils.RandomDataUtils;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by ginger on 05.11.15.
 * https://st.yandex-team.ru/TESTIRT-6666
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-43826")
@Features(DynamicTextAdTargetsFeatures.ADD)
public class AddWebpagesUnitsTest {
    private static LogSteps log = LogSteps.getLogger(AddWebpagesUnitsTest.class);
    private static String client = DynamicTextAdTargetsLogins.UNITS_ADD_CLIENT;
    private static String agency = DynamicTextAdTargetsLogins.AGENCY_UNITS_ADD;
    private static String subclient = DynamicTextAdTargetsLogins.SUBCLIENT_UNITS_ADD;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(client);

    private static Long pid;
    private static Long cid;
    private static Long agencyCid;

    @Rule
    public Trashman trashman = new Trashman(api);

    private static final int ADD_COST = 20;
    private static final int ITEM_COST_ON_ADD = 5;
    private static final int ADD_FAULT_COST = 20;
    private static final int GENERAL_FAULT_COST = 50;

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        cid = api.as(client).userSteps.campaignSteps().addDefaultDynamicTextCampaign();
        agencyCid = api.as(agency).userSteps.campaignSteps().addDefaultDynamicTextCampaign(subclient);
    }

    @Before
    @Step("Сбросим использованные баллы")
    public void prepareClient() {
        api.as(client);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        pid = api.userSteps.adGroupsSteps().addDefaultGroupDynamic(cid);
    }

    @Test
    @Description("Добавление одиночного условия")
    public void addSingleWebpageTest() {
        api.as(client);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withWebpages(new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name")
                        .withBid(30000000l)
                ),
                ExpectedResult.success()
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("стоимость добавления одного условия", unitsBefore - unitsAfter,
                equalTo(ADD_COST + ITEM_COST_ON_ADD));
    }

    @Test
    @Description("Добавление нескольких условий")
    public void addSeveralWebpagesTest() {
        api.as(client);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withWebpages(
                        new WebpageAddItemMap()
                                .withAdGroupId(pid)
                                .withName("Name1")
                                .withConditions(
                                        new WebpageConditionMap()
                                                .withOperand(WebpageConditionOperandEnum.PAGE_CONTENT)
                                                .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                                .withArguments("картоха")
                                )
                                .withBid(30000000l),
                        new WebpageAddItemMap()
                                .withAdGroupId(pid)
                                .withName("Name2")
                                .withConditions(
                                        new WebpageConditionMap()
                                                .withOperand(WebpageConditionOperandEnum.PAGE_CONTENT)
                                                .withOperator(StringConditionOperatorEnum.NOT_CONTAINS_ALL)
                                                .withArguments("ginger")
                                )
                                .withBid(30000000l)
                ),
                ExpectedResult.success(),
                ExpectedResult.success()
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость добавления двух условий", unitsBefore - unitsAfter,
                equalTo(ADD_COST + ITEM_COST_ON_ADD * 2));
    }

    @Test
    @Description("Добавление одного невалидного условия")
    public void addInvalidWebpageItemTest() {
        api.as(client);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        JavaOrPerlExpectedResult expectedResult = JavaOrPerlExpectedResult
                .java(ExpectedResult.errors(
                        new Notification(5004, Api5ErrorDetails.INVALID_FORMAT_WEBPAGE_CONDITION, 1)
                                .withDetails("Неправильный формат правила №1: задано неподдерживаемое отношение"))
                )
                .perl(ExpectedResult.errors(
                        new Notification(5004, Api5ErrorDetails.INVALID_FORMAT_WEBPAGE_CONDITION, 1))
                );

        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap()
                        .withWebpages(
                                new WebpageAddItemMap()
                                        .withConditions(
                                                new WebpageConditionMap()
                                                        .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                                        .withOperator(StringConditionOperatorEnum.EQUALS_ANY)
                                                        .withArguments(RandomDataUtils.getRandomString(10))
                                        )
                                        .withName("Name1")
                                        .withBid(30000000l)
                                        .withAdGroupId(pid)
                        ),
                expectedResult);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при добавлении условия", unitsBefore - unitsAfter,
                equalTo(ADD_COST + ADD_FAULT_COST));
    }

    @Test
    @Description("Добавление двух невалидных условий")
    public void addFewInvalidWebpagesTest() {
        api.as(client);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        JavaOrPerlExpectedResult expectedResult1 = JavaOrPerlExpectedResult
                .java(ExpectedResult.errors(
                        new Notification(5004, Api5ErrorDetails.INVALID_FORMAT_WEBPAGE_CONDITION, 1)
                                .withDetails("Неправильный формат правила №1: задано неподдерживаемое отношение")))
                .perl(ExpectedResult.errors(
                        new Notification(5004, Api5ErrorDetails.INVALID_FORMAT_WEBPAGE_CONDITION, 1)));

        JavaOrPerlExpectedResult expectedResult2 = JavaOrPerlExpectedResult
                .java(ExpectedResult.errors(
                        new Notification(8800, Api5ErrorDetails.GROUP_NOT_FOUND)
                                .withDetails("Группа объявлений не найдена"))
                )
                .perl(ExpectedResult.errors(
                        new Notification(8800, Api5ErrorDetails.GROUP_NOT_FOUND))
                );

        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap()
                        .withWebpages(
                                new WebpageAddItemMap()
                                        .withConditions(
                                                new WebpageConditionMap()
                                                        .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                                        .withOperator(StringConditionOperatorEnum.EQUALS_ANY)
                                                        .withArguments(RandomDataUtils.getRandomString(10))
                                        )
                                        .withName("Name1")
                                        .withBid(30000000l)
                                        .withAdGroupId(pid),
                                new WebpageAddItemMap()
                                        .withConditions(
                                                new WebpageConditionMap()
                                                        .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                                        .withOperator(StringConditionOperatorEnum.NOT_CONTAINS_ALL)
                                                        .withArguments(RandomDataUtils.getRandomString(10))
                                        )
                                        .withName("Name1")
                                        .withBid(30000000l)
                                        .withAdGroupId(1l)
                        ),
                expectedResult1,
                expectedResult2);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при добавлении двух условий", unitsBefore - unitsAfter,
                equalTo(ADD_COST + ADD_FAULT_COST * 2));
    }

    @Test
    @Description("Добавление одного корректного и одного невалидного условия")
    public void addOneValidOneInvalidWebpagesTest() {
        api.as(client);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        JavaOrPerlExpectedResult expectedResult = JavaOrPerlExpectedResult
                .java(ExpectedResult.errors(
                        new Notification(5004, Api5ErrorDetails.INVALID_FORMAT_WEBPAGE_CONDITION, 1)
                                .withDetails("Неправильный формат правила №1: задано неподдерживаемое отношение")))
                .perl(ExpectedResult.errors(
                        new Notification(5004, Api5ErrorDetails.INVALID_FORMAT_WEBPAGE_CONDITION, 1)));

        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap()
                        .withWebpages(
                                new WebpageAddItemMap()
                                        .withConditions(
                                                new WebpageConditionMap()
                                                        .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                                        .withOperator(StringConditionOperatorEnum.EQUALS_ANY)
                                                        .withArguments(RandomDataUtils.getRandomString(10))
                                        )
                                        .withName("Name1")
                                        .withBid(30000000l)
                                        .withAdGroupId(pid),
                                new WebpageAddItemMap()
                                        .withConditions(
                                                new WebpageConditionMap()
                                                        .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                                        .withOperator(StringConditionOperatorEnum.NOT_CONTAINS_ALL)
                                                        .withArguments(RandomDataUtils.getRandomString(10))
                                        )
                                        .withName("Name1")
                                        .withBid(30000000l)
                                        .withAdGroupId(pid)
                        ),
                expectedResult,
                JavaOrPerlExpectedResult.both(ExpectedResult.success())
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при добавлении нескольких условий",
                unitsBefore - unitsAfter, equalTo(ADD_COST + ADD_FAULT_COST + ITEM_COST_ON_ADD));
    }

    @Test
    @Description("Общая ошибка при добавлении условия")
    public void generalErrorOnAddWebpageTest() {
        api.as(client);

        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.dynamicTextAdTargetsSteps().expectErrorOnDynamicTextAdTargetsAdd(
                new AddRequestMap().withWebpages(),
                new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_NOT_LESS_ELEMENTS,
                        capitalize(AddRequestMap.WEBPAGES), 1));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);

        assertThat("стоимость ошибки при добавлении условия", unitsBefore - unitsAfter, equalTo(GENERAL_FAULT_COST));
    }

    @Test
    @Description("Недостаточное колличество баллов при добавлнии условия")
    public void notEnoughUnitsOnAddWebpageTest() {
        api.as(client);
        log.info("Обнулим колличество доступных пользователю баллов");
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(client, unitsBefore);
        unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assumeThat("доступные баллы пользователя", unitsBefore, equalTo(0));
        api.userSteps.dynamicTextAdTargetsSteps().expectErrorOnDynamicTextAdTargetsAdd(
                new AddRequestMap().withWebpages(new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name")
                        .withBid(30000000l)
                ),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));

        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(client);
        assertThat("доступные баллы пользователя после ошибки", unitsAfter, equalTo(0));
    }

    @Test
    @Description("Добавление агентством нескольких условий")
    public void addWebpagesUnitsCostByAgencyTest() {
        api.as(agency);
        pid = api.userSteps.adGroupsSteps().addDefaultGroupDynamic(agencyCid, subclient);
        int agencyUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        int scUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withWebpages(
                        new WebpageAddItemMap()
                                .withAdGroupId(pid)
                                .withName("Name1")
                                .withConditions(
                                        new WebpageConditionMap()
                                                .withOperand(WebpageConditionOperandEnum.PAGE_CONTENT)
                                                .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                                .withArguments("картоха")
                                )
                                .withBid(30000000l)
                ),
                subclient,
                ExpectedResult.success()
        );
        int scUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        int agencyUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);

        assertThat("списание баллов у агентства", agencyUnitsBefore - agencyUnitsAfter, equalTo(0));
        assertThat("списание баллов у субклиента", scUnitsBefore - scUnitsAfter, equalTo(ADD_COST + ITEM_COST_ON_ADD));
    }

    @Test
    @Description("Списание баллов у агентства при вызове метода без указания логина субклиента")
    public void unknownClientUnitsCostByAgencyTest() {
        api.as(agency);
        int agencyUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);

        api.userSteps.dynamicTextAdTargetsSteps().expectErrorOnDynamicTextAdTargetsAdd(
                new AddRequestMap().withWebpages(),
                new Api5Error(8000, Api5ErrorDetails.NOT_CLIENT_IN_CLIENT_LOGIN));
        int agencyUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);

        assertThat("списание баллов у агентства", agencyUnitsBefore - agencyUnitsAfter, equalTo(GENERAL_FAULT_COST));
    }
}
