package ru.yandex.autotests.directapi.dynamictextadtargets.delete;

import java.util.List;

import com.yandex.direct.api.v5.dynamictextadtargets.StringConditionOperatorEnum;
import com.yandex.direct.api.v5.dynamictextadtargets.WebpageConditionOperandEnum;
import org.apache.commons.lang3.StringUtils;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsFeatures;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsLogins;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.DeleteRequestMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageAddItemMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageConditionMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageGetItemMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by ginger on 06.11.15.
 * https://st.yandex-team.ru/TESTIRT-6666
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-43826")
@Features(DynamicTextAdTargetsFeatures.DELETE)
@Description("Проверка списания баллов при удалении условий нацеливания")
public class DeleteUnitsTest {
    private static final String LOGIN = DynamicTextAdTargetsLogins.UNITS_DELETE_CLIENT;
    private static int shard;
    private static Long cid;
    private static Long pid;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final int CALL_COST = 10;
    private static final int WEBPAGE_COST = 1;
    private static final int ERROR_COST = 20;
    private static final int COMMON_REQUEST_ERROR_COST = 50;

    @BeforeClass
    public static void initTest() {
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);

        cid = api.as(LOGIN).userSteps.campaignSteps().addDefaultDynamicTextCampaign();
        pid = api.userSteps.adGroupsSteps().addDefaultGroupDynamic(cid);
    }

    @Before
    public void addUnitsToClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        api.as(LOGIN);
    }

    @Test
    @Description("Удаление одного условия")
    public void deleteOneWebpage() {
        long webpageID = api.userSteps.dynamicTextAdTargetsSteps().addWebpage(
                new WebpageAddItemMap()
                                .withAdGroupId(pid)
                                .withName("Name")
                                .withBid(30000000l)
        );
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsDelete(
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap().withIds(webpageID))
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + WEBPAGE_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Удаление двух условий")
    public void deleteTwoWebpages() {
        List<Long> webpageIDs = api.userSteps.dynamicTextAdTargetsSteps().addWebpages(
                new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name")
                        .withConditions(new WebpageConditionMap()
                                .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                .withArguments("ginger")
                        )
                        .withBid(30000000l),
                new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name2")
                        .withConditions(new WebpageConditionMap()
                                .withOperand(WebpageConditionOperandEnum.PAGE_CONTENT)
                                .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                .withArguments("ginger")
                        )
                        .withBid(30000000l)
        );
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsDelete(
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap().withIds(webpageIDs.get(0), webpageIDs.get(1)))
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + WEBPAGE_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Удаление одного невалидного условия")
    public void deleteOneInvalidWebpage() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnDelete(
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(0l)),
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetails.FIELD_MUST_BE_POSITIVE_INTEGER,
                        StringUtils.capitalize(WebpageGetItemMap.ID))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Удаление двух невалидных условий")
    public void deleteTwoInvalidWebpages() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnDelete(
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(0l, 1l)),
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetails.FIELD_MUST_BE_POSITIVE_INTEGER,
                        StringUtils.capitalize(WebpageGetItemMap.ID))),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.WEBPAGE_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Удаление одного валидного и одного невалидного условия")
    public void deleteOneValidAndOneInvalidWebpages() {
        long webpageID = api.userSteps.dynamicTextAdTargetsSteps().addWebpage(
                new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name")
                        .withConditions(new WebpageConditionMap()
                                .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                .withArguments("картоха")
                        )
                        .withBid(30000000l)
        );
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnDelete(
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(webpageID, 0l)),
                ExpectedResult.success(webpageID),
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetails.FIELD_MUST_BE_POSITIVE_INTEGER,
                        StringUtils.capitalize(WebpageGetItemMap.ID))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST + WEBPAGE_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Невалидный запрос на удаление условия")
    public void invalidRequest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().expectErrorOnDynamicTextAdTargetsDelete(
                new DeleteRequestMap()
                        .withSelectionCriteria(null),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        capitalize(DeleteRequestMap.SELECTION_CRITERIA)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = COMMON_REQUEST_ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Удаление условия, когда у клиента нету баллов")
    public void deleteWebpageWhenClientHasZeroUnits() {
        long webpageID = api.userSteps.dynamicTextAdTargetsSteps().addWebpage(
                new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name")
                        .withConditions(new WebpageConditionMap()
                                .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                .withArguments("ginger")
                        )
                        .withBid(30000000l)
        );
        int units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(LOGIN, units);
        units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assumeThat("у клиента нету баллов", units, equalTo(0));

        api.userSteps.dynamicTextAdTargetsSteps().expectErrorOnDynamicTextAdTargetsDelete(
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(webpageID)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
    }
}
