package ru.yandex.autotests.directapi.dynamictextadtargets.get;

import java.util.Arrays;
import java.util.List;

import com.yandex.direct.api.v5.dynamictextadtargets.GetResponse;
import com.yandex.direct.api.v5.dynamictextadtargets.StringConditionOperatorEnum;
import com.yandex.direct.api.v5.dynamictextadtargets.WebpageConditionOperandEnum;
import com.yandex.direct.api.v5.dynamictextadtargets.WebpageFieldEnum;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsFeatures;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsLogins;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageAddItemMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageConditionMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpagesSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.LimitOffsetMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.extractProperty;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by ginger on 26.10.15.
 * https://st.yandex-team.ru/TESTIRT-6666
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-43826")
@Features(DynamicTextAdTargetsFeatures.GET)
@Description("Проверка логики полей Limit и Offset в Page")
public class GetPageTest {
    private static String client = DynamicTextAdTargetsLogins.SINGLE_CLIENT_GET;
    private static Long cid;
    private static Long pid;
    private static Long[] webpageIdsOfPid;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(client);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);

        cid = api.userSteps.campaignSteps().addDefaultDynamicTextCampaign();
        pid = api.userSteps.adGroupsSteps().addDefaultGroupDynamic(cid);
        webpageIdsOfPid = api.userSteps.dynamicTextAdTargetsSteps().addWebpages(
                new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withConditions(new WebpageConditionMap()
                                .withOperand(WebpageConditionOperandEnum.PAGE_CONTENT)
                                .withOperator(StringConditionOperatorEnum.NOT_CONTAINS_ALL)
                                .withArguments("ginger")
                        )
                        .withBid(30000000l)
                        .withName("Name1"),
                new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withConditions(new WebpageConditionMap()
                                .withOperand(WebpageConditionOperandEnum.PAGE_CONTENT)
                                .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                .withArguments("direct")
                        )
                        .withBid(30000000l)
                        .withName("Name2"),
                new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withConditions(new WebpageConditionMap()
                                .withOperand(WebpageConditionOperandEnum.PAGE_CONTENT)
                                .withOperator(StringConditionOperatorEnum.NOT_CONTAINS_ALL)
                                .withArguments("картоха")
                        )
                        .withBid(30000000l)
                        .withName("Name3"),
                new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withConditions(new WebpageConditionMap()
                                .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                .withArguments("october")
                        )
                        .withBid(30000000l)
                        .withName("Name4")
        ).toArray(new Long[]{});
    }

    @Issue("DIRECT-48819")
    @Test
    public void getHead() {
        GetResponse getResponse = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(
                new GetRequestMap().withSelectionCriteria(
                        new WebpagesSelectionCriteriaMap()
                                .withIds(webpageIdsOfPid)
                )
                        .withFieldNames(WebpageFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(2l)
                                        .withOffset(0l)));
        List<Long> gotids = extractProperty(getResponse.getWebpages(), WebpageFieldEnum.ID.value());
        assertThat("вернулся правильный набор Webpages", gotids, equalTo(Arrays.asList(webpageIdsOfPid[0], webpageIdsOfPid[1])));
    }

    @Issue("DIRECT-48819")
    @Test
    public void getBody() {
        GetResponse getResponse = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(
                new GetRequestMap().withSelectionCriteria(
                        new WebpagesSelectionCriteriaMap()
                                .withIds(webpageIdsOfPid)
                )
                        .withFieldNames(WebpageFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(2l)
                                        .withOffset(1l)));
        List<Long> gotids = extractProperty(getResponse.getWebpages(), WebpageFieldEnum.ID.value());
        assertThat("вернулся правильный набор Webpages", gotids, equalTo(Arrays.asList(webpageIdsOfPid[1], webpageIdsOfPid[2])));
    }

    @Test
    public void getTail() {
        GetResponse getResponse = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(
                new GetRequestMap().withSelectionCriteria(
                        new WebpagesSelectionCriteriaMap()
                                .withIds(webpageIdsOfPid)
                )
                        .withFieldNames(WebpageFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(2l)
                                        .withOffset(2l)));
        List<Long> gotids = extractProperty(getResponse.getWebpages(), WebpageFieldEnum.ID.value());
        assertThat("вернулся правильный набор Webpages", gotids, equalTo(Arrays.asList(webpageIdsOfPid[2], webpageIdsOfPid[3])));
    }

    @Test
    public void getPartOfTail() {
        GetResponse getResponse = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(
                new GetRequestMap().withSelectionCriteria(
                        new WebpagesSelectionCriteriaMap()
                                .withIds(webpageIdsOfPid)
                )
                        .withFieldNames(WebpageFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(2l)
                                        .withOffset(3l)));
        List<Long> gotids = extractProperty(getResponse.getWebpages(), WebpageFieldEnum.ID.value());
        assertThat("вернулся правильный набор Webpages", gotids, equalTo(Arrays.asList(webpageIdsOfPid[3])));
    }

    @Test
    public void getOverTail() {
        GetResponse getResponse = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(
                new GetRequestMap().withSelectionCriteria(
                        new WebpagesSelectionCriteriaMap()
                                .withIds(webpageIdsOfPid)
                )
                        .withFieldNames(WebpageFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(2l)
                                        .withOffset(4l)));
        assertThat("вернулся правильный набор Webpages", getResponse.getWebpages(), hasSize(0));
    }

    @Issue("DIRECT-48819")
    @Test
    public void onlyLimit() {
        GetResponse getResponse = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(
                new GetRequestMap().withSelectionCriteria(
                        new WebpagesSelectionCriteriaMap()
                                .withIds(webpageIdsOfPid)
                )
                        .withFieldNames(WebpageFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(2l)));
        List<Long> gotids = extractProperty(getResponse.getWebpages(), WebpageFieldEnum.ID.value());
        assertThat("вернулся правильный набор Webpages", gotids, equalTo(Arrays.asList(webpageIdsOfPid[0], webpageIdsOfPid[1])));
    }

    @Test
    public void onlyOffset() {
        GetResponse getResponse = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(
                new GetRequestMap().withSelectionCriteria(
                        new WebpagesSelectionCriteriaMap()
                                .withIds(webpageIdsOfPid)
                )
                        .withFieldNames(WebpageFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withOffset(2l)));
        List<Long> gotids = extractProperty(getResponse.getWebpages(), WebpageFieldEnum.ID.value());
        assertThat("вернулся правильный набор Webpages", gotids, equalTo(Arrays.asList(webpageIdsOfPid[2], webpageIdsOfPid[3])));
    }

    @Test
    public void zeroLimit() {
        api.userSteps.dynamicTextAdTargetsSteps().expectErrorOnDynamicTextAdTargetsGet(
                new GetRequestMap().withSelectionCriteria(
                        new WebpagesSelectionCriteriaMap()
                                .withIds(webpageIdsOfPid)
                )
                        .withFieldNames(WebpageFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(0l)),
                new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0));
    }

    @Test
    public void negativeLimit() {
        api.userSteps.dynamicTextAdTargetsSteps().expectErrorOnDynamicTextAdTargetsGet(
                new GetRequestMap().withSelectionCriteria(
                        new WebpagesSelectionCriteriaMap()
                                .withIds(webpageIdsOfPid)
                )
                        .withFieldNames(WebpageFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(-1l)),
                new Api5Error(4002, Api5ErrorDetails.LIMIT_MUST_BE_MORE_THAN_0));
    }

    @Test
    public void maxLimit() {
        GetResponse getResponse = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(
                new GetRequestMap().withSelectionCriteria(
                        new WebpagesSelectionCriteriaMap()
                                .withIds(webpageIdsOfPid)
                )
                        .withFieldNames(WebpageFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(LimitOffsetMap.LIMIT_MAX_LENGTH)));
        List<Long> gotids = extractProperty(getResponse.getWebpages(), WebpageFieldEnum.ID.value());
        assertThat("вернулся правильный набор Webpages", gotids, equalTo(Arrays.asList(webpageIdsOfPid)));
    }

    @Test
    public void overMaxLimit() {
        api.userSteps.dynamicTextAdTargetsSteps().expectErrorOnDynamicTextAdTargetsGet(
                new GetRequestMap().withSelectionCriteria(
                        new WebpagesSelectionCriteriaMap()
                                .withIds(webpageIdsOfPid)
                )
                        .withFieldNames(WebpageFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(LimitOffsetMap.LIMIT_MAX_LENGTH + 1)),
                new Api5Error(4002, Api5ErrorDetails.LIMIT_CANNOT_BE_MORE, LimitOffsetMap.LIMIT_MAX_LENGTH));
    }

    @Test
    public void negativeOffset() {
        api.userSteps.dynamicTextAdTargetsSteps().expectErrorOnDynamicTextAdTargetsGet(
                new GetRequestMap().withSelectionCriteria(
                        new WebpagesSelectionCriteriaMap()
                                .withIds(webpageIdsOfPid)
                )
                        .withFieldNames(WebpageFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withOffset(-1l)),
                new Api5Error(4002, Api5ErrorDetails.OFFSET_MUST_BE_MORE_OR_EQUAL_THAN_0));
    }

    @Test
    public void checkLimitedByIfLimitDidntWork() {
        GetResponse getResponse = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(
                new GetRequestMap().withSelectionCriteria(
                        new WebpagesSelectionCriteriaMap()
                                .withIds(webpageIdsOfPid)
                )
                        .withFieldNames(WebpageFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(4l)));
        assertThat("значение LimitedBy верно", getResponse.getLimitedBy(), nullValue());
    }

    @Test
    public void checkLimitedByIfLimitWorked() {
        GetResponse getResponse = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(
                new GetRequestMap().withSelectionCriteria(
                        new WebpagesSelectionCriteriaMap()
                                .withIds(webpageIdsOfPid)
                )
                        .withFieldNames(WebpageFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withOffset(1l)
                                        .withLimit(2l)));
        assertThat("значение LimitedBy верно", getResponse.getLimitedBy(), equalTo(3l));
    }

    @Test
    public void useLimitAndNotExistWebPages() {
        long nonExistWebpagesId = webpageIdsOfPid[0] + 1000;
        GetResponse getResponse = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(
                new GetRequestMap().withSelectionCriteria(
                        new WebpagesSelectionCriteriaMap()
                                .withIds(webpageIdsOfPid[0], webpageIdsOfPid[1], nonExistWebpagesId)
                )
                        .withFieldNames(WebpageFieldEnum.ID)
                        .withPage(
                                new LimitOffsetMap()
                                        .withLimit(2l)));
        List<Long> gotids = extractProperty(getResponse.getWebpages(), WebpageFieldEnum.ID.value());
        assertThat("вернулся правильный набор Webpages", gotids, equalTo(Arrays.asList(webpageIdsOfPid[0], webpageIdsOfPid[1])));
    }

}
