package ru.yandex.autotests.directapi.dynamictextadtargets.resume;

import com.yandex.direct.api.v5.dynamictextadtargets.StringConditionOperatorEnum;
import com.yandex.direct.api.v5.dynamictextadtargets.WebpageConditionOperandEnum;
import org.apache.commons.lang3.StringUtils;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsFeatures;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsLogins;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.ResumeRequestMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageAddItemMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageConditionMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageGetItemMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by ginger on 11.11.15.
 * https://st.yandex-team.ru/TESTIRT-6666
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-43826")
@Features(DynamicTextAdTargetsFeatures.RESUME)
@Description("Проверка списания баллов при включении условия нацеливания")
public class ResumeUnitsTest {

    private static final String LOGIN = DynamicTextAdTargetsLogins.UNITS_RESUME_CLIENT;
    private static Long cid;
    private static Long pid;
    private static int shard;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final int CALL_COST = 10;
    private static final int WEBPAGE_COST = 1;
    private static final int ERROR_COST = 20;
    private static final int COMMON_REQUEST_ERROR_COST = 50;

    @BeforeClass
    public static void initTest() {
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        cid = api.as(LOGIN).userSteps.campaignSteps().addDefaultDynamicTextCampaign(LOGIN);
    }

    @Before
    public void addUnitsToClient() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        pid = api.as(LOGIN).userSteps.adGroupsSteps().addDefaultGroupDynamic(cid, LOGIN);
    }

    @Test
    @Description("Включение одного условия")
    public void resumeOneWebpage() {
        long webpageID = createAndSuspendWebpage(WebpageConditionOperandEnum.PAGE_TITLE);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsResume(
                new ResumeRequestMap().withSelectionCriteria(new IdsCriteriaMap().withIds(webpageID))
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + WEBPAGE_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Включение двух условий")
    public void resumeTwoWebpages() {
        long firstWebpageID = createAndSuspendWebpage(WebpageConditionOperandEnum.PAGE_TITLE);
        long secondWebpageID = createAndSuspendWebpage(WebpageConditionOperandEnum.PAGE_CONTENT);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);

        api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsResume(
                new ResumeRequestMap().withSelectionCriteria(new IdsCriteriaMap().withIds(firstWebpageID, secondWebpageID))
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + WEBPAGE_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Включение одного невалидного условия")
    public void resumeOneInvalidWebpage() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnResume(
                new ResumeRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(0l)),
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetails.FIELD_MUST_BE_POSITIVE_INTEGER,
                        StringUtils.capitalize(WebpageGetItemMap.ID))));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Включение двух невалидных условий")
    public void resumeTwoInvalidWebpages() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnResume(
                new ResumeRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(0l, 1l)),
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetails.FIELD_MUST_BE_POSITIVE_INTEGER,
                        StringUtils.capitalize(WebpageGetItemMap.ID))),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.WEBPAGE_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST * 2;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Включение одного валидного и одного невалидного условия")
    public void resumeOneValidAndOneInvalidWebpages() {
        long webpageID = createAndSuspendWebpage(WebpageConditionOperandEnum.PAGE_TITLE);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnResume(
                new ResumeRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(webpageID, 1l)),
                ExpectedResult.success(webpageID),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.WEBPAGE_NOT_FOUND)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = CALL_COST + ERROR_COST + WEBPAGE_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Невалидный запрос на включение условий")
    public void invalidRequest() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.dynamicTextAdTargetsSteps().expectErrorOnDynamicTextAdTargetsResume(
                new ResumeRequestMap()
                        .withSelectionCriteria(null),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        StringUtils.capitalize(ResumeRequestMap.SELECTION_CRITERIA)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        int actualCost = unitsBefore - unitsAfter;
        int expectedCost = COMMON_REQUEST_ERROR_COST;
        assertThat("списалось верное число баллов", actualCost, equalTo(expectedCost));
    }

    @Test
    @Description("Включение условия, когда у клиента нету баллов")
    public void resumeWhenClientHasZeroUnits() {
        long webpageID = createAndSuspendWebpage(WebpageConditionOperandEnum.PAGE_TITLE);

        int units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(LOGIN, units);
        units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(LOGIN);
        assumeThat("у клиента нету баллов", units, equalTo(0));

        api.userSteps.dynamicTextAdTargetsSteps().expectErrorOnDynamicTextAdTargetsResume(
                new ResumeRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(webpageID)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION));
    }

    private static long createAndSuspendWebpage(WebpageConditionOperandEnum operand) {
        long webpageID = api.userSteps.dynamicTextAdTargetsSteps().addWebpage(
                new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name")
                        .withConditions(
                                new WebpageConditionMap()
                                .withArguments("картоха")
                                .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                .withOperand(operand)
                        )
                        .withBid(30000000l)
        );
        api.userSteps.getDirectJooqDbSteps().useShard(shard).bidsDynamicSteps().setSuspended(pid, webpageID);
        return webpageID;
    }
}
