package ru.yandex.autotests.directapi.dynamictextadtargets.resume;

import java.util.List;

import com.yandex.direct.api.v5.dynamictextadtargets.AddResponse;
import com.yandex.direct.api.v5.dynamictextadtargets.StringConditionOperatorEnum;
import com.yandex.direct.api.v5.dynamictextadtargets.WebpageConditionOperandEnum;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.BidsDynamicRecord;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.BannerFakeInfo;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.CampaignFakeInfo;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.GroupFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsFeatures;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsLogins;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.DeleteRequestMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.ResumeRequestMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageAddItemMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageConditionMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by ginger on 11.11.15.
 * https://st.yandex-team.ru/TESTIRT-6666
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-43826")
@Features(DynamicTextAdTargetsFeatures.RESUME)
@Description("Проверка у группы времени последнего изменения и статуса синхронизации с БК при возобновлении условия нацеливания")
public class ResumeWebpagesAdGroupLastChangeAndBsSyncedTest {
    private static LogSteps log = LogSteps.getLogger(ResumeWebpagesAdGroupLastChangeAndBsSyncedTest.class);
    private static final String LOGIN = DynamicTextAdTargetsLogins.SINGLE_CLIENT_RESUME;

    private static DateTime lastChange = DateTime.now().minusDays(5);
    private static final DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss");
    private static int shard;
    private static Long pid;
    private static AddResponse response;
    private static Long cid;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);
    private static Long adId;
    private static Long adId2;
    private static List<BidsDynamicRecord> webpages;
    private static Long adGroupId;

    @Rule
    public Trashman trashman = new Trashman(api);

    @BeforeClass
    public static void prepareGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        cid = api.userSteps.campaignSteps().addDefaultDynamicTextCampaign(LOGIN);
        adGroupId = api.userSteps.adGroupsSteps().addDefaultGroupDynamic(cid);
        adId = api.userSteps.adsSteps().addDefaultDynamicTextAd(adGroupId);
        adId2 = api.userSteps.adsSteps().addDefaultDynamicTextAd(adGroupId);
        BannerFakeInfo bannerParams = api.userSteps.bannersFakeSteps().getBannerParams(adId);
        pid = bannerParams.getPid().longValue();
        shard = api.userSteps.clientFakeSteps().getUserShard(LOGIN);
        List<Long> ids = api.userSteps.getDirectJooqDbSteps().useShard(shard)
                .bidsDynamicSteps()
                .getBidsDynamicCondIdsByPid(pid);
        for(int i=0;i<ids.size();i++){
            api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsDelete(new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap().withIds(ids.get(i))));}
        log.info("Добавим в группу условие");
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnAdd(
                new ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.AddRequestMap().withWebpages(new WebpageAddItemMap()
                        .withAdGroupId(pid)
                        .withName("Name")
                        .withBid(30000000l)
                        .withConditions(new WebpageConditionMap()
                                .withOperand(WebpageConditionOperandEnum.OFFERS_LIST_URL)
                                .withOperator(StringConditionOperatorEnum.EQUALS_ANY)
                                .withArguments("https://ginger.com")
                        )),
                ExpectedResult.success()
        );
        webpages = api.userSteps.getDirectJooqDbSteps().bidsDynamicSteps().getBidsDynamicByPid(pid);

        api.userSteps.campaignFakeSteps().makeCampaignModerated(cid);
        api.userSteps.groupFakeSteps().makeGroupFullyModerated(pid);
        api.userSteps.bannersFakeSteps().makeBannersModerated(adId);
        api.userSteps.bannersFakeSteps().makeBannersModerated(adId2);

        log.info("Выставим статусы синхронизации с БК фейково");
        api.userSteps.campaignFakeSteps().setBSSynced(cid, true);
        api.userSteps.groupFakeSteps().setGroupFakeStatusBsSynced(pid, Status.YES);
        api.userSteps.getDirectJooqDbSteps().bidsDynamicSteps().setSuspended(pid, webpages.get(0).getDynCondId());
        api.userSteps.bannersFakeSteps().setStatusBsSynced(adId, Status.YES);
        api.userSteps.bannersFakeSteps().setStatusBsSynced(adId2, Status.YES);

        log.info("Сдвинем время последнего изменения на несколько дней");
        api.userSteps.campaignFakeSteps().setLastChange(cid, lastChange.toString());
        api.userSteps.groupFakeSteps().setLastChange(pid, lastChange.toString());
        api.userSteps.bannersFakeSteps().setLastChange(adId, lastChange.toString());
        api.userSteps.bannersFakeSteps().setLastChange(adId2, lastChange.toString());
        GroupFakeInfo groupInfo = api.userSteps.groupFakeSteps().getGroupParams(pid);

        log.info("Возобновим условие нацеливания");
        api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsResume(
                new ResumeRequestMap().withSelectionCriteria(new IdsCriteriaMap().withIds(
                        webpages.get(0).getDynCondId()
                ))
        );
    }

    @Test
    public void campaignStatusBsSyncedTest() {
        CampaignFakeInfo campaignInfo = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(cid);

        assertThat("статус синхронизации с БК у кампании после возобновления условия нацеливания",
                campaignInfo.getStatusBsSynced(), equalTo(Status.YES));
    }

    @Test
    public void groupStatusBsSyncedTest() {
        GroupFakeInfo groupInfo = api.userSteps.groupFakeSteps().getGroupParams(pid);

        assertThat("статус синхронизации с БК у группы после возобновления условия нацеливания",
                groupInfo.getStatusBsSynced(), equalTo(Status.NO));
    }

    @Test
    public void bannerStatusBsSyncedTest() {
        BannerFakeInfo bannerInfo = api.userSteps.bannersFakeSteps().getBannerParams(adId);

        assertThat("статус синхронизации с БК у баннера после возобновления ключевого слова",
                bannerInfo.getStatusBsSynced(), equalTo(Status.NO));
        bannerInfo = api.userSteps.bannersFakeSteps().getBannerParams(adId2);

        assertThat("статус синхронизации с БК у баннера после возобновления ключевого слова",
                bannerInfo.getStatusBsSynced(), equalTo(Status.NO));
    }

    @Test
    public void campaignLastChangeTest() {
        CampaignFakeInfo campaignInfo = api.userSteps.campaignFakeSteps().fakeGetCampaignParams(cid);
        assumeThat("время последнего изменения кампании", campaignInfo.getLastChange(), notNullValue());
        LocalDate date = formatter.parseLocalDate(campaignInfo.getLastChange());

        assertThat("время последнего изменения кампании после возобновления условия нацеливания",
                date, equalTo(lastChange.toLocalDate()));
    }

    @Test
    public void groupLastChangeTest() {
        GroupFakeInfo groupInfo = api.userSteps.groupFakeSteps().getGroupParams(pid);
        assumeThat("время последнего изменения группы", groupInfo.getLastChange(), notNullValue());
        LocalDate date = formatter.parseLocalDate(groupInfo.getLastChange());

        assertThat("время последнего изменения группы после возобновления условия нацеливания",
                date, equalTo(LocalDate.now()));
    }

    @Test
    public void bannerLastChangeTest() {
        BannerFakeInfo bannerInfo = api.userSteps.bannersFakeSteps().getBannerParams(adId);
        assumeThat("время последнего изменения кампании", bannerInfo.getLastChange(), notNullValue());
        LocalDate date = formatter.parseLocalDate(bannerInfo.getLastChange());

        assertThat("время последнего изменения баннера после возобновления ключевого слова", date, equalTo(lastChange.toLocalDate()));
        bannerInfo = api.userSteps.bannersFakeSteps().getBannerParams(adId2);
        assumeThat("время последнего изменения кампании", bannerInfo.getLastChange(), notNullValue());
        date = formatter.parseLocalDate(bannerInfo.getLastChange());

        assertThat("время последнего изменения баннера после возобновления ключевого слова", date, equalTo(lastChange.toLocalDate()));
    }
}
