package ru.yandex.autotests.directapi.dynamictextadtargets.setbids;

import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.List;
import java.util.Queue;
import java.util.Random;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.dynamictextadtargets.GetResponse;
import com.yandex.direct.api.v5.dynamictextadtargets.StringConditionOperatorEnum;
import com.yandex.direct.api.v5.dynamictextadtargets.WebpageConditionOperandEnum;
import com.yandex.direct.api.v5.dynamictextadtargets.WebpageFieldEnum;
import com.yandex.direct.api.v5.dynamictextadtargets.WebpageGetItem;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsFeatures;
import ru.yandex.autotests.directapi.dynamictextadtargets.DynamicTextAdTargetsLogins;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.SetBidsItemMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.SetBidsRequestMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageAddItemMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageConditionMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpageGetItemMap;
import ru.yandex.autotests.directapi.model.api5.dynamictextadtargets.WebpagesSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.SetBidsExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.CompareStrategy;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategies;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.isIn;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;
import static ru.yandex.autotests.irt.testutils.beandiffer2.beanfield.BeanFieldPath.newPath;

/**
 * Created by semkagtn on 10.11.15.
 * https://st.yandex-team.ru/TESTIRT-7700
 */
@Aqua.Test
@Issue("https://st.yandex-team.ru/DIRECT-43826")
@Features(DynamicTextAdTargetsFeatures.SET_BIDS)
@Description("Установка значения Bid методом SetBids")
public class SetBidTest {

    private static final String LOGIN = DynamicTextAdTargetsLogins.SINGLE_CLIENT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Long campaignId;

    private static Long firstAdGroupId;
    private static Long firstWebpageId1;
    private static Long firstWebpageId2;

    private static Long secondAdGroupId;
    private static Long secondWebpageId;


    private static Long campaignIdElse;

    private static Long adGroupIdElse;
    private static Long webpageIdElse;

    @BeforeClass
    public static void prepareStaticData() {
        campaignId = api.userSteps.campaignSteps().addDefaultDynamicTextCampaign();
        firstAdGroupId = api.userSteps.adGroupsSteps().addDefaultGroupDynamic(campaignId);
        secondAdGroupId = api.userSteps.adGroupsSteps().addDefaultGroupDynamic(campaignId);

        campaignIdElse = api.userSteps.campaignSteps().addDefaultDynamicTextCampaign();
        adGroupIdElse = api.userSteps.adGroupsSteps().addDefaultGroupDynamic(campaignIdElse);

        List<Long> webpageIds = api.userSteps.dynamicTextAdTargetsSteps().addWebpages(
                new WebpageAddItemMap()
                        .withAdGroupId(firstAdGroupId)
                        .withName("Name1")
                        .withConditions(new WebpageConditionMap()
                                .withOperand(WebpageConditionOperandEnum.PAGE_TITLE)
                                .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                .withArguments("arg1"))
                        .withBid(15_000_000L),
                new WebpageAddItemMap()
                        .withAdGroupId(firstAdGroupId)
                        .withName("Name2")
                        .withConditions(new WebpageConditionMap()
                                .withOperand(WebpageConditionOperandEnum.PAGE_CONTENT)
                                .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                .withArguments("arg2"))
                        .withBid(15_000_000L),
                new WebpageAddItemMap()
                        .withAdGroupId(secondAdGroupId)
                        .withName("Name3")
                        .withConditions(new WebpageConditionMap()
                                .withOperand(WebpageConditionOperandEnum.PAGE_CONTENT)
                                .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                .withArguments("arg3"))
                        .withBid(15_000_000L),
                new WebpageAddItemMap()
                        .withAdGroupId(adGroupIdElse)
                        .withName("Name4")
                        .withConditions(new WebpageConditionMap()
                                .withOperand(WebpageConditionOperandEnum.PAGE_CONTENT)
                                .withOperator(StringConditionOperatorEnum.CONTAINS_ANY)
                                .withArguments("arg4"))
                        .withBid(15_000_000L));
        firstWebpageId1 = webpageIds.get(0);
        firstWebpageId2 = webpageIds.get(1);
        secondWebpageId = webpageIds.get(2);
        webpageIdElse = webpageIds.get(3);
    }

    @Before
    public void clearSpentUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
    }

    private static Queue<Long> bids = new Random().longs(20, 70)
            .mapToObj(x -> x * 1_000_000L)
            .distinct()
            .limit(10)
            .collect(Collectors.toCollection(ArrayDeque::new));

    public static Long getNewBid() {
        return bids.remove();
    }

    @Test
    @Description("Установить ставку на поиске по идентификатору Webpage")
    public void setBidById() {
        Long bid = getNewBid();
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap().withBids(
                new SetBidsItemMap()
                        .withId(firstWebpageId1)
                        .withBid(bid)),
                SetBidsExpectedResult.successWithId());

        GetResponse response = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(new GetRequestMap()
                .withFieldNames(WebpageFieldEnum.ID, WebpageFieldEnum.BID)
                .withSelectionCriteria(new WebpagesSelectionCriteriaMap()
                        .withCampaignIds(campaignId, campaignIdElse)));
        List<WebpageGetItem> actualResult = response.getWebpages();
        actualResult.sort((x, y) -> x.getId().compareTo(y.getId()));

        CompareStrategy strategy = DefaultCompareStrategies.onlyExpectedFields()
                .forFields(newPath("[1-3]", WebpageGetItemMap.BID)).useMatcher(not(equalTo(bid)));
        List<WebpageGetItem> expectedResult = Arrays.asList(
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(bid)
                        .withId(firstWebpageId1).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(0L)
                        .withId(firstWebpageId2).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(0L)
                        .withId(secondWebpageId).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(0L)
                        .withId(webpageIdElse).getBean());

        assertThat("ставки на поиске изменились верно", actualResult, beanDiffer(expectedResult)
                .useCompareStrategy(strategy));
    }

    @Test
    @Description("Установить ставку на поиске по двум идентификаторам Webpage")
    public void setBidByTwoIds() {
        Long firstBid = getNewBid();
        Long secondBid = getNewBid();
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap().withBids(
                new SetBidsItemMap()
                        .withId(firstWebpageId1)
                        .withBid(firstBid),
                new SetBidsItemMap()
                        .withId(secondWebpageId)
                        .withBid(secondBid)),
                SetBidsExpectedResult.successWithId(), SetBidsExpectedResult.successWithId());

        GetResponse response = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(new GetRequestMap()
                .withFieldNames(WebpageFieldEnum.ID, WebpageFieldEnum.BID)
                .withSelectionCriteria(new WebpagesSelectionCriteriaMap()
                        .withCampaignIds(campaignId, campaignIdElse)));
        List<WebpageGetItem> actualResult = response.getWebpages();
        actualResult.sort((x, y) -> x.getId().compareTo(y.getId()));

        CompareStrategy strategy = DefaultCompareStrategies.onlyExpectedFields()
                .forFields(newPath("1|3", WebpageGetItemMap.BID)).useMatcher(not(isIn(Arrays.asList(firstBid, secondBid))));
        List<WebpageGetItem> expectedResult = Arrays.asList(
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(firstBid)
                        .withId(firstWebpageId1).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(0L)
                        .withId(firstWebpageId2).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(secondBid)
                        .withId(secondWebpageId).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(0L)
                        .withId(webpageIdElse).getBean());

        assertThat("ставки на поиске изменились верно", actualResult, beanDiffer(expectedResult)
                .useCompareStrategy(strategy));
    }

    @Test
    @Description("Установить ставку на поиске по идентификатору группы")
    public void setBidByAdGroupId() {
        Long bid = getNewBid();
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap().withBids(
                new SetBidsItemMap()
                        .withAdGroupId(firstAdGroupId)
                        .withBid(bid)),
                SetBidsExpectedResult.successWithAdGroupId());

        GetResponse response = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(new GetRequestMap()
                .withFieldNames(WebpageFieldEnum.ID, WebpageFieldEnum.BID)
                .withSelectionCriteria(new WebpagesSelectionCriteriaMap()
                        .withCampaignIds(campaignId, campaignIdElse)));
        List<WebpageGetItem> actualResult = response.getWebpages();
        actualResult.sort((x, y) -> x.getId().compareTo(y.getId()));

        CompareStrategy strategy = DefaultCompareStrategies.onlyExpectedFields()
                .forFields(newPath("[2-3]", WebpageGetItemMap.BID)).useMatcher(not(equalTo(bid)));
        List<WebpageGetItem> expectedResult = Arrays.asList(
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(bid)
                        .withId(firstWebpageId1).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(bid)
                        .withId(firstWebpageId2).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(0L)
                        .withId(secondWebpageId).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(0L)
                        .withId(webpageIdElse).getBean());

        assertThat("ставки на поиске изменились верно", actualResult, beanDiffer(expectedResult)
                .useCompareStrategy(strategy));
    }

    @Test
    @Description("Установить ставку на поиске по идентификаторам двух групп")
    public void setBidByTwoAdGroupIds() {
        Long firstBid = getNewBid();
        Long secondBid = getNewBid();
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap().withBids(
                new SetBidsItemMap()
                        .withAdGroupId(firstAdGroupId)
                        .withBid(firstBid),
                new SetBidsItemMap()
                        .withAdGroupId(secondAdGroupId)
                        .withBid(secondBid)),
                SetBidsExpectedResult.successWithAdGroupId(), SetBidsExpectedResult.successWithAdGroupId());

        GetResponse response = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(new GetRequestMap()
                .withFieldNames(WebpageFieldEnum.ID, WebpageFieldEnum.BID)
                .withSelectionCriteria(new WebpagesSelectionCriteriaMap()
                        .withCampaignIds(campaignId, campaignIdElse)));
        List<WebpageGetItem> actualResult = response.getWebpages();
        actualResult.sort((x, y) -> x.getId().compareTo(y.getId()));

        CompareStrategy strategy = DefaultCompareStrategies.onlyExpectedFields()
                .forFields(newPath("3", WebpageGetItemMap.BID)).useMatcher(not(isIn(Arrays.asList(firstBid, secondBid))));
        List<WebpageGetItem> expectedResult = Arrays.asList(
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(firstBid)
                        .withId(firstWebpageId1).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(firstBid)
                        .withId(firstWebpageId2).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(secondBid)
                        .withId(secondWebpageId).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(0L)
                        .withId(webpageIdElse).getBean());

        assertThat("ставки на поиске изменились верно", actualResult, beanDiffer(expectedResult)
                .useCompareStrategy(strategy));
    }

    @Test
    @Description("Установить ставку на поиске по идентификатору кампании")
    public void setBidByCampaignId() {
        Long bid = getNewBid();
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap().withBids(
                new SetBidsItemMap()
                        .withCampaignId(campaignId)
                        .withBid(bid)),
                SetBidsExpectedResult.successWithCampaignId());

        GetResponse response = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(new GetRequestMap()
                .withFieldNames(WebpageFieldEnum.ID, WebpageFieldEnum.BID)
                .withSelectionCriteria(new WebpagesSelectionCriteriaMap()
                        .withCampaignIds(campaignId, campaignIdElse)));
        List<WebpageGetItem> actualResult = response.getWebpages();
        actualResult.sort((x, y) -> x.getId().compareTo(y.getId()));

        CompareStrategy strategy = DefaultCompareStrategies.onlyExpectedFields()
                .forFields(newPath("3", WebpageGetItemMap.BID)).useMatcher(not(equalTo(bid)));
        List<WebpageGetItem> expectedResult = Arrays.asList(
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(bid)
                        .withId(firstWebpageId1).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(bid)
                        .withId(firstWebpageId2).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(bid)
                        .withId(secondWebpageId).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(0L)
                        .withId(webpageIdElse).getBean());

        assertThat("ставки на поиске изменились верно", actualResult, beanDiffer(expectedResult)
                .useCompareStrategy(strategy));
    }

    @Test
    @Description("Установить ставку на поиске по двум идентификаторам кампаний")
    public void setBidsByTwoCampaignIds() {
        Long firstBid = getNewBid();
        Long secondBid = getNewBid();
        api.userSteps.dynamicTextAdTargetsSteps().shouldGetResultOnSetBids(new SetBidsRequestMap().withBids(
                new SetBidsItemMap()
                        .withCampaignId(campaignId)
                        .withBid(firstBid),
                new SetBidsItemMap()
                        .withCampaignId(campaignIdElse)
                        .withBid(secondBid)),
                SetBidsExpectedResult.successWithCampaignId(), SetBidsExpectedResult.successWithCampaignId());

        GetResponse response = api.userSteps.dynamicTextAdTargetsSteps().dynamicTextAdTargetsGet(new GetRequestMap()
                .withFieldNames(WebpageFieldEnum.ID, WebpageFieldEnum.BID)
                .withSelectionCriteria(new WebpagesSelectionCriteriaMap()
                        .withCampaignIds(campaignId, campaignIdElse)));
        List<WebpageGetItem> actualResult = response.getWebpages();
        actualResult.sort((x, y) -> x.getId().compareTo(y.getId()));

        List<WebpageGetItem> expectedResult = Arrays.asList(
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(firstBid)
                        .withId(firstWebpageId1).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(firstBid)
                        .withId(firstWebpageId2).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(firstBid)
                        .withId(secondWebpageId).getBean(),
                (WebpageGetItem) new WebpageGetItemMap()
                        .withBid(secondBid)
                        .withId(webpageIdElse).getBean());

        assertThat("ставки на поиске изменились верно", actualResult, beanDiffer(expectedResult));
    }
}
